import { SystemHealth, SystemAlert, SystemLog, HealthCheck, DashboardStats, MetricHistoryPoint, ApiResponse } from '../types';

// Configure your backend URL here
const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090/api/system_monitoring';

// Helper to get auth headers
const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    ...(token && { 'Authorization': `Bearer ${token}` })
  };
};

export class SystemMonitoringAPI {
  static async getSystemHealth(): Promise<SystemHealth> {
    const response = await fetch(`${API_BASE_URL}/health`, { headers: getAuthHeaders() });
    const data: ApiResponse<SystemHealth> = await response.json();
    if (data.status === 'error') throw new Error(data.message);
    return data.data!;
  }

  static async getMetricsHistory(type: string, hours: number = 24): Promise<MetricHistoryPoint[]> {
    const response = await fetch(`${API_BASE_URL}/metrics/history?type=${type}&hours=${hours}`, { headers: getAuthHeaders() });
    const data: ApiResponse<MetricHistoryPoint[]> = await response.json();
    if (data.status === 'error') throw new Error(data.message);
    return data.data!;
  }

  static async getSystemAlerts(filters: any = {}): Promise<{ data: SystemAlert[], pagination: any }> {
    const params = new URLSearchParams(filters);
    const response = await fetch(`${API_BASE_URL}/alerts?${params}`, { headers: getAuthHeaders() });
    const data: ApiResponse<SystemAlert[]> = await response.json();
    if (data.status === 'error') throw new Error(data.message);
    return { data: data.data!, pagination: data.pagination };
  }

  static async getSystemLogs(filters: any = {}): Promise<{ data: SystemLog[], pagination: any }> {
    const params = new URLSearchParams(filters);
    const response = await fetch(`${API_BASE_URL}/logs?${params}`, { headers: getAuthHeaders() });
    const data: ApiResponse<SystemLog[]> = await response.json();
    if (data.status === 'error') throw new Error(data.message);
    return { data: data.data!, pagination: data.pagination };
  }

  static async getHealthChecks(): Promise<HealthCheck[]> {
    const response = await fetch(`${API_BASE_URL}/health-checks`, { headers: getAuthHeaders() });
    const data: ApiResponse<HealthCheck[]> = await response.json();
    if (data.status === 'error') throw new Error(data.message);
    return data.data!;
  }

  static async getDashboardStats(): Promise<DashboardStats> {
    const response = await fetch(`${API_BASE_URL}/dashboard/stats`, { headers: getAuthHeaders() });
    const data: ApiResponse<DashboardStats> = await response.json();
    if (data.status === 'error') throw new Error(data.message);
    return data.data!;
  }

  static async acknowledgeAlert(alertId: number): Promise<void> {
    const response = await fetch(`${API_BASE_URL}/alerts/${alertId}/acknowledge`, {
      method: 'POST',
      headers: getAuthHeaders()
    });
    const data = await response.json();
    if (data.status === 'error') throw new Error(data.message);
  }

  static async resolveAlert(alertId: number): Promise<void> {
    const response = await fetch(`${API_BASE_URL}/alerts/${alertId}/resolve`, {
      method: 'POST',
      headers: getAuthHeaders()
    });
    const data = await response.json();
    if (data.status === 'error') throw new Error(data.message);
  }
}