// Integration and Performance Tests for Loan Calculator

import { loanCalculatorAPI } from '../api/loanCalculator';
import { formatInterestRate, validateInterestRate, formatProcessingFee } from '../utils/dataStandardization';

// Integration Tests
export const integrationTests = {
  testAPIConnection: async (): Promise<boolean> => {
    try {
      const isHealthy = await loanCalculatorAPI.healthCheck();
      console.log('✅ API Health Check:', isHealthy ? 'PASS' : 'FAIL');
      return isHealthy;
    } catch (error) {
      console.error('❌ API Health Check: FAIL', error);
      return false;
    }
  },

  testBankRatesAPI: async (): Promise<boolean> => {
    try {
      const startTime = performance.now();
      const response = await loanCalculatorAPI.getBankRates({
        state: 'karnataka',
        city: 'bangalore'
      });
      const endTime = performance.now();
      
      const responseTime = endTime - startTime;
      console.log(`✅ Bank Rates API: ${responseTime.toFixed(2)}ms`);
      
      const isValid = response.success && 
                     Array.isArray(response.data.banks) &&
                     response.data.banks.length > 0;
      
      return isValid && responseTime < 2000;
    } catch (error) {
      console.error('❌ Bank Rates API: FAIL', error);
      return false;
    }
  }
};

// Performance Tests
export const performanceTests = {
  testEMICalculationSpeed: (): boolean => {
    const iterations = 1000;
    const startTime = performance.now();
    
    for (let i = 0; i < iterations; i++) {
      const principal = 5000000;
      const rate = 8.5;
      const tenure = 20;
      
      const monthlyRate = rate / 12 / 100;
      const numPayments = tenure * 12;
      const emi = (principal * monthlyRate * Math.pow(1 + monthlyRate, numPayments)) / 
                  (Math.pow(1 + monthlyRate, numPayments) - 1);
    }
    
    const endTime = performance.now();
    const avgTime = (endTime - startTime) / iterations;
    
    console.log(`✅ EMI Calculation Speed: ${avgTime.toFixed(4)}ms per calculation`);
    return avgTime < 0.1;
  }
};

// Accuracy Tests
export const accuracyTests = {
  testInterestRateFormatting: (): boolean => {
    const testCases = [
      { input: 8.756, expected: 8.76 },
      { input: 9.001, expected: 9.00 }
    ];
    
    let passed = 0;
    testCases.forEach(({ input, expected }) => {
      const result = formatInterestRate(input);
      if (result === expected) {
        passed++;
        console.log(`✅ Rate formatting: ${input} -> ${result}`);
      } else {
        console.error(`❌ Rate formatting: ${input} -> ${result}, expected ${expected}`);
      }
    });
    
    return passed === testCases.length;
  }
};

export const runAllTests = async (): Promise<void> => {
  console.log('🧪 Starting Loan Calculator Tests...\n');
  
  console.log('📡 Integration Tests:');
  await integrationTests.testAPIConnection();
  await integrationTests.testBankRatesAPI();
  
  console.log('\n⚡ Performance Tests:');
  performanceTests.testEMICalculationSpeed();
  
  console.log('\n🎯 Accuracy Tests:');
  accuracyTests.testInterestRateFormatting();
  
  console.log('\n✅ All tests completed!');
};