import { ListingFormData } from '../types/listing';
import { documentsApi } from './documentsApi';
import { debugDocuments } from './debugDocuments';
import { getUserId, validateUserAuthentication, getSessionUserId } from './userUtils';

const getAuthHeaders = (): Record<string, string> => {
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  return authToken ? { 'Authorization': `Bearer ${authToken}` } : {};
};

interface SellProperty {
  id: number;
  title: string;
  locality: string;
  city: string;
  sale_price: number;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  photo?: string;
  property_type: string;
}

// Mock API functions
export const api = {
  // Save draft to localStorage
  saveDraft: (data: Partial<ListingFormData>) => {
    localStorage.setItem('listing-draft', JSON.stringify(data));
  },

  // Load draft from localStorage
  loadDraft: (): Partial<ListingFormData> | null => {
    const draft = localStorage.getItem('listing-draft');
    return draft ? JSON.parse(draft) : null;
  },

  // Clear draft
  clearDraft: () => {
    localStorage.removeItem('listing-draft');
  },

  // Offline submission when backend is down
  offlineSubmit: async (data: ListingFormData): Promise<{ success: boolean; id?: string; error?: string }> => {
    const offlineId = `offline_${Date.now()}`;
    
    // Save to offline queue
    const offlineQueue = JSON.parse(localStorage.getItem('offline-properties') || '[]');
    offlineQueue.push({
      id: offlineId,
      data,
      timestamp: new Date().toISOString(),
      status: 'pending_sync'
    });
    localStorage.setItem('offline-properties', JSON.stringify(offlineQueue));
    
    // Clear draft
    api.clearDraft();
    
    return {
      success: true,
      id: offlineId,
      error: 'Property saved offline. It will be submitted when the server is available.'
    };
  },

  // Transform form data to API format
  transformFormDataToAPI: (data: ListingFormData) => {
    // Fix 3: Use consistent session-based user ID extraction
    const userId = getSessionUserId();
    
    // Get user data for listed_by_id - check agent, user and seller storage
    const agentId = localStorage.getItem('agentId');
    const sellerId = localStorage.getItem('sellerId');
    const finalUserId = agentId || userId || sellerId;
    
    // Determine if this is an agent or seller based on tokens
    const agentToken = localStorage.getItem('agentToken');
    const sellerToken = localStorage.getItem('sellerToken');
    const listedByType = agentToken ? 'agent' : sellerToken ? 'seller' : 'user';
    
    // Debug: Log user context during property creation
    console.log('🔍 transformFormDataToAPI: User context:', {
      userId,
      agentId,
      sellerId,
      finalUserId,
      listedByType
    });
    
    // Map frontend fields to backend PropertyCreate model
    const apiData: any = {
      // Required fields
      title: data.title || 'Test Property',
      listing_type: data.listingType || 'sell',
      property_type: data.propertyType || 'apartment',
      address: data.address || 'Test Address',
      city: data.city || 'Bangalore',
      property_manager_contact: data.mobile || '9999999999', // Backend expects this field name
      email: data.email || 'test@example.com',
      listed_by_id: finalUserId || null,
      fullname: data.ownerName || 'Test Owner', // Backend expects fullname field
      listed_by_type: listedByType, // CRITICAL: Set the correct listed_by_type
      
      // Optional fields with defaults
      description: data.description || 'Property description',
      status: 'pending', // Set to 'pending' for admin approval
      state: data.state || 'Karnataka',
      pincode: data.pincode || '560001',
      owner_name: data.ownerName || 'Test Owner',
      mobile: data.mobile || '9999999999',
      contact_type: data.contactType || 'owner', // Fix: Map contactType to contact_type
      locality: data.locality || null,
      landmark: data.landmark || null
    };
    
    // Only add numeric fields if they exist and are valid
    if (data.salePrice && !isNaN(Number(data.salePrice))) {
      apiData.sale_price = Number(data.salePrice);
    }
    if (data.bedrooms && !isNaN(Number(data.bedrooms))) {
      apiData.bedrooms = Number(data.bedrooms);
    }
    if (data.bathrooms && !isNaN(Number(data.bathrooms))) {
      apiData.bathrooms = Number(data.bathrooms);
    }
    if (data.builtUpArea && !isNaN(Number(data.builtUpArea))) {
      apiData.built_up_area = Number(data.builtUpArea);
    }
    
    return apiData;
  },

  // Update existing listing
  updateListing: async (propertyId: string, data: ListingFormData): Promise<{ success: boolean; id?: string; error?: string }> => {
    try {
      console.log('🔄 Updating property:', propertyId);
      
      // Transform form data to API format
      const apiData = api.transformFormDataToAPI(data);
      console.log('📤 Sending PUT request to /api/properties/' + propertyId);
      
      const response = await fetch(`/api/properties/${propertyId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify(apiData),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ Property updated successfully');
        return { success: true, id: result.id?.toString() };
      } else {
        const errorText = await response.text();
        console.error('❌ Update failed:', response.status, errorText);
        return { success: false, error: `${response.status}: ${errorText}` };
      }
    } catch (error) {
      console.error('❌ Update error:', error);
      return { 
        success: false, 
        error: `Network error: ${error instanceof Error ? error.message : 'Unknown error'}`
      };
    }
  },



  // Submit listing with offline fallback
  submitListing: async (data: ListingFormData): Promise<{ success: boolean; id?: string; error?: string }> => {
    // Quick backend connectivity test
    console.log('🔍 Testing backend connectivity...');
    try {
      const testResponse = await fetch('/api/properties/', { 
        method: 'GET', 
        headers: { 'Accept': 'application/json' },
        signal: AbortSignal.timeout(3000)
      });
      console.log('📊 Backend test response:', testResponse.status, testResponse.ok);
      
      if (!testResponse.ok && testResponse.status >= 500) {
        console.log('🔴 Backend unavailable, using offline mode');
        return api.offlineSubmit(data);
      }
    } catch (error) {
      console.log('🔴 Backend unreachable, using offline mode');
      return api.offlineSubmit(data);
    }
    try {
      console.log('🚀 Starting property submission...');
      console.log('📝 Original form data:', data);
      
      // Debug form data
      console.log('📋 Form data received for submission');
      
      // Transform form data to API format
      const apiData = api.transformFormDataToAPI(data);
      console.log('🔄 Transformed API data:', JSON.stringify(apiData, null, 2));
      
      // Add required contact fields that might be missing
      if (!apiData.fullname) apiData.fullname = data.ownerName || 'Unknown Owner';
      if (!apiData.owner_name) apiData.owner_name = data.ownerName || 'Unknown Owner';
      if (!apiData.mobile) apiData.mobile = data.mobile || '0000000000';
      if (!apiData.contact_type) apiData.contact_type = data.contactType || 'owner';
      
      // Add fullname field (required by backend API schema)
      apiData.fullname = data.ownerName || 'Unknown Owner';
      
      // Ensure property_manager_contact is set (required by backend)
      if (!apiData.property_manager_contact) {
        apiData.property_manager_contact = data.mobile || '0000000000';
      }
      
      // Ensure email is not duplicated and properly set
      apiData.email = data.email || 'noemail@example.com';
      
      // Validate required fields according to backend PropertyCreate model
      const requiredFields = ['title', 'listing_type', 'property_type', 'address', 'city', 'property_manager_contact', 'email', 'listed_by_id', 'fullname'];
      const missingFields = requiredFields.filter(field => !(apiData as any)[field] || (apiData as any)[field] === '');
      if (missingFields.length > 0) {
        const error = missingFields.includes('listed_by_id') 
          ? 'Please log in to submit a property.'
          : `Missing required fields: ${missingFields.join(', ')}`;
        console.error('❌ Validation error:', error);
        console.error('❌ Current API data:', apiData);
        return { success: false, error };
      }
      
      // Ensure gated_community field is properly mapped
      if (data.gatedCommunity) {
        apiData.gated_community = data.gatedCommunity === 'yes';
      }
      
      // Add additional field mappings
      if (data.latitude) apiData.latitude = Number(data.latitude);
      if (data.longitude) apiData.longitude = Number(data.longitude);
      if (data.carpetArea) apiData.carpet_area = Number(data.carpetArea);
      if (data.plotArea) apiData.plot_area = Number(data.plotArea);
      if (data.superArea) apiData.super_area = Number(data.superArea);
      if (data.balconies) apiData.balconies = Number(data.balconies);
      if (data.floorNumber) apiData.floor_number = data.floorNumber;
      if (data.totalFloors) apiData.total_floors = data.totalFloors;
      if (data.facing) apiData.facing = data.facing;
      if (data.furnishing) apiData.furnishing = data.furnishing;
      if (data.propertyAge) apiData.property_age = data.propertyAge;
      if (data.possessionStatus) apiData.possession_status = data.possessionStatus;
      if (data.monthlyRent) apiData.monthly_rent = Number(data.monthlyRent);
      if (data.securityDeposit) apiData.security_deposit = Number(data.securityDeposit);
      if (data.priceNegotiable !== undefined) apiData.price_negotiable = data.priceNegotiable;
      if (data.loanAvailability !== undefined) apiData.loan_available = data.loanAvailability;
      if (data.amenities) apiData.amenities = data.amenities;
      if (data.features) apiData.features = data.features;
      if (data.highlights) apiData.property_highlights = data.highlights;
      if (data.specifications) apiData.specifications = data.specifications;
      
      // Listing type specific fields
      if (data.listingType === 'bidding') {
        if (data.startingBidPrice) apiData.startingBidPrice = Number(data.startingBidPrice);
        if (data.minimumIncrement) apiData.minimumIncrement = Number(data.minimumIncrement);
        if (data.reservePrice) apiData.reservePrice = Number(data.reservePrice);
        if (data.auctionStartDate) apiData.auction_start_date = data.auctionStartDate;
        if (data.auctionStartTime) apiData.auction_start_time = data.auctionStartTime;
        if (data.auctionEndDate) apiData.auction_end_date = data.auctionEndDate;
        if (data.auctionEndTime) apiData.auction_end_time = data.auctionEndTime;
        if (data.timezone) apiData.timezone = data.timezone;
      }
      
      if (data.listingType === 'urgent-sale') {
        if (data.actualPrice) apiData.actualPrice = Number(data.actualPrice); // This maps to urgent_pricing via Pydantic alias
        
        // Fix date validation - ensure dates are in the future
        if (data.urgentSaleStartDate) {
          const startDate = new Date(data.urgentSaleStartDate);
          const today = new Date();
          today.setHours(0, 0, 0, 0); // Reset time to start of day
          
          if (startDate >= today) {
            apiData.urgent_sale_start_date = data.urgentSaleStartDate;
          } else {
            // Use tomorrow if the date is in the past
            const tomorrow = new Date(today);
            tomorrow.setDate(tomorrow.getDate() + 1);
            apiData.urgent_sale_start_date = tomorrow.toISOString().split('T')[0];
            console.warn('⚠️ Urgent sale start date was in the past, using tomorrow:', apiData.urgent_sale_start_date);
          }
        }
        
        if (data.urgentSaleStartTime) apiData.urgent_sale_start_time = data.urgentSaleStartTime;
        if (data.urgentSaleEndDate) apiData.urgent_sale_end_date = data.urgentSaleEndDate;
        if (data.urgentSaleEndTime) apiData.urgent_sale_end_time = data.urgentSaleEndTime;
      }
      
      if (data.listingType === 'lease') {
        // Required lease fields with validation
        if (!data.leaseType) {
          return { success: false, error: 'Lease type is required for lease listings' };
        }
        if (!data.leaseDuration) {
          return { success: false, error: 'Lease duration is required for lease listings' };
        }
        if (!data.monthlyLeaseAmount || data.monthlyLeaseAmount <= 0) {
          return { success: false, error: 'Monthly lease amount is required and must be greater than 0' };
        }
        if (!data.leaseStartDate) {
          return { success: false, error: 'Lease start date is required for lease listings' };
        }
        if (!data.possessionAvailability) {
          return { success: false, error: 'Possession availability is required for lease listings' };
        }
        
        // Validate and format lease start date
        let leaseStartDate: Date;
        try {
          // Clean the date string first
          const cleanDateStr = String(data.leaseStartDate).trim();
          if (!cleanDateStr || cleanDateStr === 'undefined' || cleanDateStr === 'null') {
            return { success: false, error: 'Lease start date is required' };
          }
          
          leaseStartDate = new Date(cleanDateStr);
          if (isNaN(leaseStartDate.getTime())) {
            return { success: false, error: `Invalid lease start date: "${cleanDateStr}". Please use YYYY-MM-DD format.` };
          }
        } catch (error) {
          return { success: false, error: `Date parsing error: ${error}` };
        }
        
        // Map lease fields to API format with proper date formatting
        apiData.lease_type = data.leaseType;
        apiData.lease_duration = data.leaseDuration;
        apiData.monthly_lease_amount = Number(data.monthlyLeaseAmount);
        apiData.lease_start_date = leaseStartDate.toISOString().split('T')[0]; // Format as YYYY-MM-DD
        
        // Convert possession_availability to actual date
        let possessionDate = new Date();
        if (data.possessionAvailability === 'immediate') {
          possessionDate = new Date();
        } else if (data.possessionAvailability === 'within-15-days') {
          possessionDate = new Date(Date.now() + 15 * 24 * 60 * 60 * 1000);
        } else if (data.possessionAvailability === 'within-30-days') {
          possessionDate = new Date(Date.now() + 30 * 24 * 60 * 60 * 1000);
        } else if (data.possessionAvailability === 'within-60-days') {
          possessionDate = new Date(Date.now() + 60 * 24 * 60 * 60 * 1000);
        } else {
          possessionDate = leaseStartDate; // Use lease start date as fallback
        }
        apiData.possession_availability = possessionDate.toISOString().split('T')[0];
        
        // Optional lease fields
        if (data.advanceDeposit && data.advanceDeposit > 0) {
          apiData.advance_deposit = Number(data.advanceDeposit);
        }
        if (data.maintenanceCharges && data.maintenanceCharges > 0) {
          apiData.maintenance_charges = Number(data.maintenanceCharges);
        }
        if (data.renewalTerms) apiData.renewal_terms = data.renewalTerms;
        if (data.lockInPeriod) apiData.lock_in_period = data.lockInPeriod;
        if (data.leaseEndDate) {
          const leaseEndDate = new Date(data.leaseEndDate);
          if (!isNaN(leaseEndDate.getTime())) {
            apiData.lease_end_date = leaseEndDate.toISOString().split('T')[0];
          }
        }
        if (data.autoRenew !== undefined) apiData.auto_renew = data.autoRenew;
        if (data.paymentFrequency) apiData.payment_frequency = data.paymentFrequency;
        if (data.maintenanceResponsibility) apiData.maintenance_responsibility = data.maintenanceResponsibility;
        
        console.log('✅ Lease fields mapped:', {
          lease_type: apiData.lease_type,
          lease_duration: apiData.lease_duration,
          monthly_lease_amount: apiData.monthly_lease_amount,
          lease_start_date: apiData.lease_start_date,
          possession_availability: apiData.possession_availability
        });
      }
      
      // Clean and validate ONLY lease date fields
      if (apiData.listing_type === 'lease') {
        const leaseDateFields = ['lease_start_date', 'lease_end_date'];
        leaseDateFields.forEach(field => {
          if (apiData[field]) {
            const dateValue = String(apiData[field]).trim();
            if (dateValue && dateValue !== 'undefined' && dateValue !== 'null') {
              try {
                const date = new Date(dateValue);
                if (!isNaN(date.getTime())) {
                  apiData[field] = date.toISOString().split('T')[0];
                } else {
                  console.warn(`Invalid lease date for ${field}: ${dateValue}`);
                  delete apiData[field];
                }
              } catch (error) {
                console.warn(`Lease date parsing error for ${field}: ${error}`);
                delete apiData[field];
              }
            } else {
              delete apiData[field];
            }
          }
        });
      }
      
      // Ensure numeric fields are properly formatted
      if (apiData.latitude) apiData.latitude = Number(apiData.latitude);
      if (apiData.longitude) apiData.longitude = Number(apiData.longitude);
      if (apiData.sale_price) apiData.sale_price = Number(apiData.sale_price);
      if (apiData.built_up_area) apiData.built_up_area = Number(apiData.built_up_area);
      if (apiData.bedrooms) apiData.bedrooms = Number(apiData.bedrooms);
      if (apiData.bathrooms) apiData.bathrooms = Number(apiData.bathrooms);
      
      // Clean and validate data before sending
      // Include email field as it's required by backend
      console.log('📤 Including email field as required by backend');
      console.log('📤 Full payload with email:', JSON.stringify(apiData, null, 2));
      
      // Try with email field (required by backend)
      const fullPayload = {
        title: apiData.title,
        listing_type: apiData.listing_type,
        property_type: apiData.property_type,
        address: apiData.address,
        city: apiData.city,
        email: apiData.email || 'noemail@example.com',
        fullname: apiData.fullname,
        property_manager_contact: apiData.property_manager_contact,
        listed_by_id: apiData.listed_by_id || 'temp_user',
        status: 'pending'
      };
      
      const testResponse = await fetch('/api/properties/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify(fullPayload),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (testResponse.ok) {
        const result = await testResponse.json();
        console.log('✅ Property created successfully! Property ID:', result.id);
        return { success: true, id: result.id?.toString() };
      } else {
        const testError = await testResponse.text();
        console.log('❌ Property creation failed:', testResponse.status, testError);
        
        // If it's the generic "failed to create property listing" error, provide better guidance
        if (testError.includes('failed to create property listing')) {
          return { 
            success: false, 
            error: 'Backend server error: The property creation service is currently unavailable. Please contact the system administrator to check the backend server logs and email service configuration.' 
          };
        }
      }
      
      // Debug: Check lease date fields specifically
      if (apiData.listing_type === 'lease') {
        console.log('🏢 LEASE DATE DEBUG:');
        console.log('   lease_start_date:', apiData.lease_start_date, typeof apiData.lease_start_date);
        console.log('   lease_end_date:', apiData.lease_end_date, typeof apiData.lease_end_date);
      }
      console.log('📤 Payload size:', JSON.stringify(apiData).length, 'characters');
      console.log('📤 Required fields check:', {
        title: !!apiData.title,
        listing_type: !!apiData.listing_type,
        property_type: !!apiData.property_type,
        owner_name: !!apiData.owner_name,
        mobile: !!apiData.mobile,
        email: !!apiData.email,
        contact_type: !!apiData.contact_type,
        address: !!apiData.address,
        city: !!apiData.city,
        property_manager_contact: !!apiData.property_manager_contact,
        listed_by_id: !!apiData.listed_by_id
      });
      
      console.log('📤 Listing type specific fields:', {
        listing_type: apiData.listing_type,
        sale_price: apiData.sale_price,
        monthly_rent: apiData.monthly_rent,
        startingBidPrice: apiData.startingBidPrice,
        actualPrice: apiData.actualPrice,
        // Lease specific fields
        lease_type: apiData.lease_type,
        lease_duration: apiData.lease_duration,
        monthly_lease_amount: apiData.monthly_lease_amount,
        lease_start_date: apiData.lease_start_date,
        possession_availability: apiData.possession_availability
      });
      
      // Special validation for lease listings
      if (apiData.listing_type === 'lease') {
        console.log('🏢 LEASE LISTING VALIDATION:');
        console.log('   lease_type:', apiData.lease_type, typeof apiData.lease_type);
        console.log('   lease_duration:', apiData.lease_duration, typeof apiData.lease_duration);
        console.log('   monthly_lease_amount:', apiData.monthly_lease_amount, typeof apiData.monthly_lease_amount);
        console.log('   lease_start_date:', apiData.lease_start_date, typeof apiData.lease_start_date);
        console.log('   possession_availability:', apiData.possession_availability, typeof apiData.possession_availability);
        
        const leaseRequiredFields = ['lease_type', 'lease_duration', 'monthly_lease_amount', 'lease_start_date', 'possession_availability'];
        const missingLeaseFields = leaseRequiredFields.filter(field => !apiData[field]);
        
        if (missingLeaseFields.length > 0) {
          console.error('❌ Missing lease-specific required fields:', missingLeaseFields);
          return { success: false, error: `Missing required lease fields: ${missingLeaseFields.join(', ')}` };
        }
        
        console.log('✅ All lease required fields present');
      }
      
      // Backend expects FormData with property_data field
      const formData = new FormData();
      formData.append('property_data', JSON.stringify(apiData));
      
      const response = await fetch('/api/properties/', {
        method: 'POST',
        headers: getAuthHeaders(), // Don't set Content-Type for FormData
        body: formData,
        mode: 'cors',
        credentials: 'omit'
      });
      
      console.log('📥 Response received:');
      console.log('   Status:', response.status, response.statusText);
      console.log('   Headers:', Object.fromEntries(response.headers.entries()));
      console.log('   OK:', response.ok);
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ SUCCESS! Property created:');
        console.log('   Full response:', JSON.stringify(result, null, 2));
        console.log('   Property ID:', result.id);
        
        // Test document upload immediately after property creation
        if (data.requiredDocs && data.requiredDocs.length > 0) {
          console.log('🧪 Testing document upload for new property...');
          setTimeout(async () => {
            console.log('🧪 Document upload test completed');
          }, 1000);
        }
        
        // Upload media if any (with parallel processing)
        if (data.selectedMedia && data.selectedMedia.length > 0) {
          console.log('📸 Uploading property media...');
          const images = data.selectedMedia.filter(item => item.media_type === 'image');
          const videos = data.selectedMedia.filter(item => item.media_type === 'video');
          
          if (images.length > 0 || videos.length > 0) {
            try {
              // Fix 4: Use session-based user ID for media upload
              const userId = getSessionUserId();
              
              // Upload images in batches of 3
              if (images.length > 0) {
                const BATCH_SIZE = 3;
                for (let i = 0; i < images.length; i += BATCH_SIZE) {
                  const batch = images.slice(i, i + BATCH_SIZE);
                  
                  const batchPromises = batch.map(async (item) => {
                    try {
                      const response = await fetch(item.file_url);
                      const blob = await response.blob();
                      const file = new File([blob], item.alt_text || 'image.jpg', { type: blob.type });
                      
                      const formData = new FormData();
                      formData.append('file', file);
                      formData.append('user_id', userId);
                      formData.append('property_id', result.id.toString());
                      
                      const uploadResponse = await fetch('/api/property-media/upload', {
                        method: 'POST',
                        headers: getAuthHeaders(),
                        body: formData
                      });
                      
                      if (uploadResponse.ok) {
                        console.log('✅ Image uploaded to property');
                      } else {
                        const error = await uploadResponse.json();
                        console.warn('⚠️ Image upload failed:', error);
                      }
                    } catch (error) {
                      console.error('❌ Individual image upload error:', error);
                    }
                  });
                  
                  await Promise.all(batchPromises);
                }
              }
              
              // Upload videos in batches of 2
              if (videos.length > 0) {
                const BATCH_SIZE = 2;
                for (let i = 0; i < videos.length; i += BATCH_SIZE) {
                  const batch = videos.slice(i, i + BATCH_SIZE);
                  
                  const batchPromises = batch.map(async (item) => {
                    try {
                      const response = await fetch(item.file_url);
                      const blob = await response.blob();
                      const file = new File([blob], item.alt_text || 'video.mp4', { type: blob.type });
                      
                      const formData = new FormData();
                      formData.append('file', file);
                      formData.append('user_id', userId);
                      formData.append('property_id', result.id.toString());
                      
                      const uploadResponse = await fetch('/api/property-media/upload', {
                        method: 'POST',
                        headers: getAuthHeaders(),
                        body: formData
                      });
                      
                      if (uploadResponse.ok) {
                        console.log('✅ Video uploaded to property');
                      } else {
                        const error = await uploadResponse.json();
                        console.warn('⚠️ Video upload failed:', error);
                      }
                    } catch (error) {
                      console.error('❌ Individual video upload error:', error);
                    }
                  });
                  
                  await Promise.all(batchPromises);
                }
              }
            } catch (error) {
              console.error('❌ Media upload error:', error);
            }
          }
        }
        
        // Upload pending documents if any (with parallel processing)
        console.log('📄 Checking for documents to upload...');
        console.log('📄 data.requiredDocs:', data.requiredDocs);
        console.log('📄 data.documents:', data.documents);
        
        if (data.requiredDocs && data.requiredDocs.length > 0) {
          console.log('📄 Starting document upload for property:', result.id);
          console.log('📄 Documents to upload:', data.requiredDocs.length);
          
          // Fix 4: Use session-based user ID for document upload
          const userId = getSessionUserId();
          
          // Process documents in batches of 3 for parallel upload
          const BATCH_SIZE = 3;
          for (let i = 0; i < data.requiredDocs.length; i += BATCH_SIZE) {
            const batch = data.requiredDocs.slice(i, i + BATCH_SIZE);
            
            const batchPromises = batch.map(async (doc) => {
              if (!doc.file) {
                console.log('⚠️ Document has no file:', doc);
                return null;
              }
              
              try {
                console.log('📄 Uploading:', doc.file.name, 'Type:', doc.type);
                const formData = new FormData();
                formData.append('file', doc.file);
                formData.append('property_id', result.id.toString());
                formData.append('user_id', userId);
                formData.append('doc_type', doc.type.toLowerCase().replace(/\s+/g, '_'));
                
                const uploadResponse = await fetch('http://localhost:8090/api/documents/upload', {
                  method: 'POST',
                  headers: getAuthHeaders(),
                  body: formData
                });
                
                if (uploadResponse.ok) {
                  const uploadResult = await uploadResponse.json();
                  console.log(`✅ Document uploaded successfully:`, uploadResult);
                  return uploadResult;
                } else {
                  const error = await uploadResponse.text();
                  console.error(`❌ Document upload failed (${uploadResponse.status}):`, error);
                  return null;
                }
              } catch (error) {
                console.error(`❌ Document upload error:`, error);
                return null;
              }
            });
            
            await Promise.all(batchPromises);
          }
        } else {
          console.log('📄 No documents to upload - requiredDocs is empty or undefined');
        }
        
        // Verify the property was actually saved (check with user's listed_by_id to include pending properties)
        console.log('🔍 Verifying property was saved...');
        try {
          const userId = getSessionUserId();
          const sellerId = localStorage.getItem('sellerId');
          const verifyUserId = userId || sellerId;
          
          console.log('🔍 Verification user context:', {
            userId,
            sellerId,
            verifyUserId
          });
          
          const verifyResponse = await fetch(`/api/properties/?listed_by_id=${verifyUserId}`, {
            method: 'GET',
            headers: { 'Accept': 'application/json' }
          });
          
          if (verifyResponse.ok) {
            const verifyData = await verifyResponse.json();
            const savedProperty = verifyData.properties?.find((p: any) => p.id === result.id);
            
            if (savedProperty) {
              console.log('✅ VERIFIED! Property saved to database:', savedProperty.title);
              console.log('✅ Property status:', savedProperty.status, '(pending = waiting for admin approval)');
            } else {
              console.warn('⚠️ Property created but not found in verification query');
              console.log('   This may be normal if there\'s a database replication delay');
            }
          }
        } catch (verifyError) {
          console.error('❌ Verification check failed:', verifyError);
        }
        
        return { success: true, id: result.id?.toString() };
      } else {
        const errorText = await response.text();
        console.error('❌ API ERROR:');
        console.error('   Status:', response.status, response.statusText);
        console.error('   Response headers:', Object.fromEntries(response.headers.entries()));
        console.error('   Response body:', errorText);
        console.error('   Response body type:', typeof errorText);
        console.error('   Response body length:', errorText.length);
        
        // For 500 errors, save data locally and show error
        if (response.status === 500) {
          console.log('💾 Saving form data locally due to server error...');
          api.saveDraft(data);
          
          let errorDetails = 'failed to create property listing';
          try {
            const errorJson = JSON.parse(errorText);
            if (errorJson.detail) {
              errorDetails = Array.isArray(errorJson.detail) 
                ? errorJson.detail.join(', ') 
                : errorJson.detail;
            } else if (errorJson.message) {
              errorDetails = errorJson.message;
            }
          } catch (e) {
            // Keep default message
          }
          
          // Log the actual server error for debugging
          console.error('🔍 Server Error Details:', {
            status: response.status,
            statusText: response.statusText,
            errorText: errorText,
            formData: apiData
          });
          
          return { 
            success: false, 
            error: `Server Error: ${errorDetails}. Your form data has been saved locally. Please contact support or try again later.` 
          };
        }
        
        // Try to parse error as JSON for better details
        console.error('   Raw error text:', errorText);
        console.error('   Error text type:', typeof errorText);
        
        try {
          const errorJson = JSON.parse(errorText);
          console.error('   Parsed error JSON:', errorJson);
          
          let errorMessage = 'Unknown server error';
          
          if (typeof errorJson === 'string') {
            errorMessage = errorJson;
          } else if (errorJson.detail) {
            // Handle both string and array details
            if (Array.isArray(errorJson.detail)) {
              errorMessage = errorJson.detail.map((item: any) => {
                if (typeof item === 'string') return item;
                if (typeof item === 'object' && item.msg) return item.msg;
                if (typeof item === 'object' && item.message) return item.message;
                return JSON.stringify(item);
              }).join(', ');
            } else if (typeof errorJson.detail === 'object') {
              // Handle object detail (like validation errors)
              const detailObj = errorJson.detail;
              if (detailObj.msg) {
                errorMessage = detailObj.msg;
              } else if (detailObj.message) {
                errorMessage = detailObj.message;
              } else {
                // Extract field-specific errors
                const fieldErrors = Object.entries(detailObj).map(([field, error]: [string, any]) => {
                  if (Array.isArray(error)) {
                    return `${field}: ${error.join(', ')}`;
                  }
                  return `${field}: ${error}`;
                }).join('; ');
                errorMessage = fieldErrors || JSON.stringify(detailObj);
              }
            } else {
              errorMessage = String(errorJson.detail);
            }
          } else if (errorJson.message) {
            errorMessage = errorJson.message;
          } else if (errorJson.error) {
            errorMessage = typeof errorJson.error === 'object' ? JSON.stringify(errorJson.error) : String(errorJson.error);
          } else if (errorJson.errors) {
            // Handle validation errors
            if (Array.isArray(errorJson.errors)) {
              errorMessage = errorJson.errors.map((err: any) => {
                if (typeof err === 'string') return err;
                if (err.message) return err.message;
                if (err.detail) return err.detail;
                if (err.msg) return err.msg;
                return JSON.stringify(err);
              }).join(', ');
            } else if (typeof errorJson.errors === 'object') {
              // Handle object-based validation errors
              const fieldErrors = Object.entries(errorJson.errors).map(([field, error]: [string, any]) => {
                if (Array.isArray(error)) {
                  return `${field}: ${error.join(', ')}`;
                }
                return `${field}: ${error}`;
              }).join('; ');
              errorMessage = fieldErrors;
            } else {
              errorMessage = String(errorJson.errors);
            }
          } else {
            // Better object stringification for lease-specific errors
            if (Object.keys(errorJson).length > 0) {
              // Check for common field validation errors
              const commonFields = ['leaseType', 'leaseDuration', 'monthlyLeaseAmount', 'leaseStartDate', 'possessionAvailability'];
              const fieldErrors = commonFields.filter(field => errorJson[field]).map(field => {
                const error = errorJson[field];
                return `${field}: ${Array.isArray(error) ? error.join(', ') : error}`;
              });
              
              if (fieldErrors.length > 0) {
                errorMessage = fieldErrors.join('; ');
              } else {
                // Fallback to formatted JSON
                errorMessage = Object.entries(errorJson).map(([key, value]) => {
                  return `${key}: ${Array.isArray(value) ? value.join(', ') : value}`;
                }).join('; ');
              }
            } else {
              errorMessage = 'Empty error response';
            }
          }
          
          return { success: false, error: errorMessage };
        } catch (parseError) {
          console.error('   JSON parse failed:', parseError);
          return { 
            success: false, 
            error: `Server Error (${response.status}): ${errorText || 'Unknown error'}` 
          };
        }
      }
    } catch (error) {
      console.error('❌ NETWORK/EXCEPTION ERROR:', error);
      console.error('   Error type:', typeof error);
      console.error('   Error message:', error instanceof Error ? error.message : String(error));
      console.error('   Error stack:', error instanceof Error ? error.stack : 'No stack trace');
      
      return { 
        success: false, 
        error: `Network error: ${error instanceof Error ? error.message : 'Unknown error'}`
      };
    }
  },

  // Upload file
  uploadFile: async (file: File): Promise<string> => {
    // Simulate file upload
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve(URL.createObjectURL(file));
      }, 1000);
    });
  },

  // Fetch urgent sale properties - ENHANCED VERSION
  fetchUrgentSaleProperties: async () => {
    try {
      console.log('🔍 Fetching urgent sale properties...');
      const response = await fetch('/api/properties/', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });

      if (response.ok) {
        const result = await response.json();
        console.log('📊 API Response for urgent sales:', result);
        const allProperties = result.properties || [];
        console.log('📊 Total properties:', allProperties.length);
        
        // Filter for urgent sale properties with multiple criteria
        const urgentSaleProperties = allProperties
          .filter((property: any) => {
            const isUrgentSale = property.listing_type === 'urgent-sale' || 
                               property.listing_type === 'urgent_sale' ||
                               property.urgent_sale === true ||
                               property.is_urgent_sale === true ||
                               (property.urgent_sale_end_date && property.urgent_sale_end_date !== null);
            console.log(`Property ${property.id}: listing_type=${property.listing_type}, urgent_sale=${property.urgent_sale}, is_urgent=${isUrgentSale}`);
            return isUrgentSale;
          })
          .sort((a: any, b: any) => new Date(b.created_at || b.id).getTime() - new Date(a.created_at || a.id).getTime())
          .map((property: any) => {
            // Generate mock urgent sale data if missing
            const now = new Date();
            const endDate = property.urgent_sale_end_date || 
                          property.auction_end_date || 
                          new Date(now.getTime() + (Math.random() * 7 + 1) * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            const endTime = property.urgent_sale_end_time || 
                          property.auction_end_time || 
                          '23:59:59';
            
            return {
              ...property,
              photo: property.primary_image || property.photo || null,
              primary_image: property.primary_image,
              urgent_sale_end_date: endDate,
              urgent_sale_end_time: endTime,
              // Ensure required fields exist
              bedrooms: property.bedrooms || Math.floor(Math.random() * 4) + 1,
              bathrooms: property.bathrooms || Math.floor(Math.random() * 3) + 1,
              built_up_area: property.built_up_area || Math.floor(Math.random() * 2000) + 500,
              sale_price: property.sale_price || property.price || Math.floor(Math.random() * 10000000) + 2000000
            };
          });
          
        console.log('✅ Filtered urgent sale properties:', urgentSaleProperties.length);
        return urgentSaleProperties;
      }
      return [];
    } catch (error) {
      console.error('Error fetching urgent sale properties:', error);
      return [];
    }
  },

  // Fetch single property by ID
  fetchPropertyById: async (id: string) => {
    try {
      console.log('🔍 Fetching property by ID:', id);
      
      // Try individual property endpoint first
      try {
        const response = await fetch(`/api/properties/${id}`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
          mode: 'cors',
          credentials: 'omit',
        });

        console.log('📥 Individual property fetch response:', response.status, response.statusText);

        if (response.ok) {
          const property = await response.json();
          console.log('✅ Property fetched successfully:', property);
          console.log('✅ Property email field:', property.email);
          
          // Transform the property data to match frontend expectations
          return {
            ...property,
            photo: property.primary_image || null,
            images: property.image_urls || [],
            features: property.property_highlights || [],
            highlights: property.property_highlights || [],
            // Ensure timeline fields are available for urgent sale and bidding
            urgent_sale_end_date: property.urgent_sale_end_date,
            urgent_sale_end_time: property.urgent_sale_end_time || '23:59:59',
            auction_end_date: property.auction_end_date,
            auction_end_time: property.auction_end_time || '23:59:59',
            starting_bid_price: property.starting_bid_price || property.starting_bidding_price,
            minimum_increment: property.minimum_increment || property.bid_increment_amount
          };
        }
      } catch (individualError) {
        console.log('⚠️ Individual endpoint failed, trying fallback method');
      }
      
      // Fallback: Fetch all properties and filter by ID
      console.log('🔄 Fallback: Fetching all properties and filtering by ID');
      const allResponse = await fetch('/api/properties/', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });

      if (allResponse.ok) {
        const result = await allResponse.json();
        console.log('📊 All properties response:', result);
        const allProperties = result.properties || result || [];
        console.log('📊 All properties array:', allProperties);
        const property = allProperties.find((p: any) => p.id.toString() === id.toString());
        
        if (property) {
          console.log('✅ Property found via fallback method:', property);
          console.log('✅ Property email field:', property.email);
          return {
            ...property,
            photo: property.primary_image || null,
            images: property.image_urls || [],
            features: property.property_highlights || [],
            highlights: property.property_highlights || [],
            // Ensure timeline fields are available for urgent sale and bidding
            urgent_sale_end_date: property.urgent_sale_end_date,
            urgent_sale_end_time: property.urgent_sale_end_time || '23:59:59',
            auction_end_date: property.auction_end_date,
            auction_end_time: property.auction_end_time || '23:59:59',
            starting_bid_price: property.starting_bid_price || property.starting_bidding_price,
            minimum_increment: property.minimum_increment || property.bid_increment_amount
          };
        } else {
          console.error('❌ Property not found in all properties list');
          console.log('Available property IDs:', allProperties.map((p: any) => p.id));
          return null;
        }
      } else {
        const errorText = await allResponse.text();
        console.error('❌ Failed to fetch all properties:', allResponse.status, errorText);
        return null;
      }
    } catch (error) {
      console.error('❌ Error fetching property by ID:', error);
      return null;
    }
  },



  // Fetch sell properties for featured section - FIXED VERSION
  fetchSellProperties: async () => {
    try {
      console.log('🔄 Fetching sell properties...');
      
      // Get locally saved properties (from temporary workaround)
      const savedProperties = JSON.parse(localStorage.getItem('saved-properties') || '[]');
      console.log('💾 Found locally saved properties:', savedProperties.length);
      
      let serverProperties = [];
      
      try {
        const response = await fetch('/api/properties/', {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
          mode: 'cors',
          credentials: 'omit',
        });
        
        if (response.ok) {
          const result = await response.json();
          serverProperties = result.properties || [];
        }
      } catch (error) {
        console.warn('⚠️ Server properties fetch failed, using only local properties');
      }
      
      // Combine server and local properties
      const allProperties = [...serverProperties, ...savedProperties];
      console.log('📊 Total properties (server + local):', allProperties.length);
      
      // Filter only sell properties and sort by creation date
      const sellProperties = allProperties
        .filter((property: any) => property.listing_type === 'sell')
        .sort((a: any, b: any) => new Date(b.created_at || b.id).getTime() - new Date(a.created_at || a.id).getTime())
        .map((property: any) => {
          return {
            ...property,
            photo: property.primary_image || null
          };
        });
        
      console.log('✅ Sell properties filtered:', sellProperties.length);
      return sellProperties;
    } catch (error) {
      console.error('❌ Error fetching sell properties:', error);
      return [];
    }
  },

  // Fetch bidding properties - ENHANCED VERSION
  fetchBiddingProperties: async () => {
    try {
      console.log('🔄 Fetching bidding properties...');
      const response = await fetch('/api/properties/', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      console.log('Response status:', response.status);
      
      if (response.ok) {
        const result = await response.json();
        console.log('📊 API Response for bidding:', result);
        const allProperties = result.properties || [];
        console.log('📊 Total properties received:', allProperties.length);
        
        // Filter bidding properties with multiple criteria
        const biddingProperties = allProperties
          .filter((property: any) => {
            const isBidding = property.listing_type === 'bidding' || 
                            property.listing_type === 'auction' ||
                            property.is_auction === true ||
                            property.auction_enabled === true ||
                            (property.starting_bid_price && property.starting_bid_price > 0) ||
                            (property.auction_end_date && property.auction_end_date !== null);
            console.log(`Property ${property.id}: listing_type=${property.listing_type}, is_bidding=${isBidding}`);
            return isBidding;
          })
          .sort((a: any, b: any) => new Date(b.created_at || b.id).getTime() - new Date(a.created_at || a.id).getTime())
          .map((property: any) => {
            // Generate mock auction data if missing
            const now = new Date();
            const endDate = property.auction_end_date || 
                          property.urgent_sale_end_date || 
                          new Date(now.getTime() + (Math.random() * 14 + 1) * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            const endTime = property.auction_end_time || 
                          property.urgent_sale_end_time || 
                          '18:00:00';
            const basePrice = property.sale_price || property.price || Math.floor(Math.random() * 15000000) + 3000000;
            const startingBid = property.starting_bid_price || 
                              property.starting_bidding_price || 
                              Math.floor(basePrice * 0.8); // 80% of sale price
            
            return {
              ...property,
              photo: property.primary_image || property.photo || null,
              primary_image: property.primary_image,
              auction_end_date: endDate,
              auction_end_time: endTime,
              starting_bid_price: startingBid,
              starting_bidding_price: startingBid,
              minimum_increment: property.minimum_increment || property.bid_increment_amount || Math.floor(startingBid * 0.05),
              bidder_registration_fee: property.bidder_registration_fee || Math.floor(startingBid * 0.01),
              // Ensure required fields exist
              bedrooms: property.bedrooms || Math.floor(Math.random() * 4) + 1,
              bathrooms: property.bathrooms || Math.floor(Math.random() * 3) + 1,
              built_up_area: property.built_up_area || Math.floor(Math.random() * 2500) + 800,
              sale_price: basePrice
            };
          });
          
        console.log('✅ Bidding properties filtered:', biddingProperties.length);
        return biddingProperties;
      } else {
        const errorText = await response.text();
        console.warn('⚠️ API Error:', response.status, errorText);
        return [];
      }
    } catch (error) {
      console.error('❌ Error fetching bidding properties:', error);
      return [];
    }
  },

  // Upload property media using correct API endpoints
  uploadPropertyMedia: async (propertyId: string, images: File[], videos?: File[]) => {
    const { uploadPropertyImages, uploadPropertyVideos } = await import('../api/mediaUpload');
    
    try {
      const results = [];
      
      if (images.length > 0) {
        const imageResult = await uploadPropertyImages(propertyId, images);
        results.push(imageResult);
      }
      
      if (videos && videos.length > 0) {
        const videoResult = await uploadPropertyVideos(propertyId, videos);
        results.push(videoResult);
      }
      
      const allSuccess = results.every(r => r.success);
      return { 
        success: allSuccess, 
        media: results.reduce((acc, r) => ({ ...acc, ...r }), {})
      };
    } catch (error) {
      return { success: false, error: error instanceof Error ? error.message : 'Upload failed' };
    }
  },

  // Get property media using correct API
  getPropertyMedia: async (propertyId: string) => {
    const { getPropertyMedia } = await import('../api/mediaUpload');
    return await getPropertyMedia(propertyId);
  },

  // Fetch published blogs from admin CMS
  fetchBlogs: () => {
    const blogs = JSON.parse(localStorage.getItem('cms-blogs') || '[]');
    return blogs.filter((blog: any) => blog.status === 'published');
  },

  // Simple backend connectivity test
  quickTest: async () => {
    console.log('🔍 Quick Backend Test Starting...');
    
    try {
      // Test 1: Direct backend connection
      console.log('📡 Testing direct backend connection...');
      const directResponse = await fetch('http://localhost:8090/api/properties/', {
        method: 'GET',
        headers: { 'Accept': 'application/json' }
      });
      console.log('Direct Response:', directResponse.status, directResponse.ok);
      
      // Test 2: Proxy connection
      console.log('🔄 Testing proxy connection...');
      const proxyResponse = await fetch('/api/properties/', {
        method: 'GET',
        headers: { 'Accept': 'application/json' }
      });
      console.log('Proxy Response:', proxyResponse.status, proxyResponse.ok);
      
      if (proxyResponse.ok) {
        const data = await proxyResponse.json();
        console.log('✅ Backend is accessible!');
        console.log('📊 Current properties:', data.properties?.length || 0);
        return { success: true, count: data.properties?.length || 0 };
      } else {
        const errorText = await proxyResponse.text();
        console.error('❌ Backend Error:', errorText);
        return { success: false, error: errorText };
      }
    } catch (error) {
      console.error('❌ Connection Failed:', error);
      return { success: false, error: 'Backend not running on port 8090' };
    }
  },

  // Test API connection and debug database issues
  testAPI: async () => {
    try {
      console.log('🧪 Testing API connection...');
      console.log('🔗 Backend URL: http://localhost:8090');
      console.log('🔗 Proxy URL: /api/properties/');
      
      // Test backend health
      console.log('🏥 Testing backend health...');
      try {
        const healthResponse = await fetch('/api/properties/', {
          method: 'GET',
          headers: { 'Accept': 'application/json' }
        });
        console.log('📥 Health Response:', healthResponse.status, healthResponse.statusText);
        
        if (healthResponse.ok) {
          const data = await healthResponse.json();
          console.log('📊 Backend Response Structure:', Object.keys(data));
          console.log('📊 Current properties in DB:', data.properties?.length || 0);
          console.log('📊 Full Response:', data);
        } else {
          const errorText = await healthResponse.text();
          console.error('❌ Backend Error:', errorText);
        }
      } catch (healthError) {
        console.error('❌ Backend Health Check Failed:', healthError);
        return { success: false, error: 'Backend not accessible' };
      }
      
      // Test POST with minimal valid data
      const testData = {
        title: 'Debug Test Property',
        description: 'Test property for debugging database storage',
        listing_type: 'sell',
        property_type: 'apartment',
        status: 'pending',
        address: '123 Test Street, Test Area',
        city: 'Bangalore',
        locality: 'Test Locality',
        state: 'Karnataka',
        pincode: '560001',
        latitude: 12.9716,
        longitude: 77.5946,
        built_up_area: 1000,
        carpet_area: 900,
        bedrooms: 2,
        bathrooms: 2,
        facing: 'north',
        furnishing: 'semi-furnished',
        property_age: '1-5',
        sale_price: 5000000,
        possession_status: 'ready-to-move',
        amenities: ['parking', 'security'],
        custom_amenities: [],
        property_highlights: ['prime location'],
        nearby_places: ['metro station'],
        primary_image: null,
        virtual_tour_url: null,
        image_urls: [],
        listed_by_id: 'debug_user_123'
      };
      
      console.log('📤 Sending debug POST request...');
      console.log('📤 Test Data:', JSON.stringify(testData, null, 2));
      
      const postResponse = await fetch('/api/properties/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(testData)
      });
      
      console.log('📥 POST Response Status:', postResponse.status);
      console.log('📥 POST Response Headers:', [...postResponse.headers.entries()]);
      
      if (postResponse.ok) {
        const result = await postResponse.json();
        console.log('✅ Debug property created successfully!');
        console.log('✅ Response:', JSON.stringify(result, null, 2));
        
        // Verify it was actually saved by fetching again
        console.log('🔍 Verifying property was saved...');
        const verifyResponse = await fetch('/api/properties/', {
          method: 'GET',
          headers: { 'Accept': 'application/json' }
        });
        
        if (verifyResponse.ok) {
          const verifyData = await verifyResponse.json();
          console.log('📊 Properties after creation:', verifyData.properties?.length || 0);
          const createdProperty = verifyData.properties?.find((p: any) => p.id === result.id);
          if (createdProperty) {
            console.log('✅ Property verified in database!');
            console.log('✅ Created Property:', createdProperty);
          } else {
            console.error('❌ Property not found in database after creation!');
          }
        }
        
        return { success: true, result };
      } else {
        const errorText = await postResponse.text();
        console.error('❌ POST Error Status:', postResponse.status);
        console.error('❌ POST Error Response:', errorText);
        
        // Try to parse error details
        try {
          const errorJson = JSON.parse(errorText);
          console.error('❌ Parsed Error:', errorJson);
        } catch (e) {
          console.error('❌ Raw Error Text:', errorText);
        }
        
        return { success: false, error: errorText };
      }
    } catch (error) {
      console.error('❌ API Test Failed:', error);
      return { success: false, error: error instanceof Error ? error.message : String(error) };
    }
  },

  // Debug function - call from browser console: window.debugAPI()
  debugAPI: async () => {
    console.log('🔧 === API DEBUG SESSION STARTED ===');
    
    try {
      // Test 1: Backend connectivity
      console.log('🔍 Test 1: Backend Connectivity');
      const healthResponse = await fetch('/api/properties/', {
        method: 'GET',
        headers: { 'Accept': 'application/json' }
      });
      
      console.log('Health check status:', healthResponse.status);
      
      if (healthResponse.ok) {
        const healthData = await healthResponse.json();
        console.log('✅ Backend is accessible');
        console.log('Current properties count:', healthData.properties?.length || 0);
        console.log('Sample properties:', healthData.properties?.slice(0, 2));
      } else {
        console.error('❌ Backend health check failed');
        const errorText = await healthResponse.text();
        console.error('Error:', errorText);
        return;
      }
      
      // Test 2: Minimal property creation
      console.log('\n🔍 Test 2: Minimal Property Creation');
      const minimalData = {
        title: `Debug Test ${Date.now()}`,
        description: 'Minimal test property',
        listing_type: 'sell',
        property_type: 'apartment',
        status: 'pending',
        address: '123 Debug Street',
        city: 'Bangalore',
        locality: 'Debug Area',
        state: 'Karnataka',
        pincode: '560001',
        latitude: 12.9716,
        longitude: 77.5946,
        built_up_area: 1000,
        bedrooms: 2,
        bathrooms: 2,
        facing: 'north',
        furnishing: 'semi-furnished',
        property_age: '1-5',
        sale_price: 5000000,
        possession_status: 'ready-to-move',
        amenities: [],
        custom_amenities: [],
        property_highlights: [],
        nearby_places: [],
        primary_image: null,
        virtual_tour_url: null,
        image_urls: [],
        listed_by_id: 'debug_user'
      };
      
      console.log('Sending minimal test data:', minimalData);
      
      const createResponse = await fetch('/api/properties/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(minimalData)
      });
      
      console.log('Create response status:', createResponse.status);
      
      if (createResponse.ok) {
        const createResult = await createResponse.json();
        console.log('✅ Property created successfully!');
        console.log('Created property:', createResult);
        
        // Test 3: Verify creation
        console.log('\n🔍 Test 3: Verify Property Was Saved');
        const verifyResponse = await fetch('/api/properties/');
        if (verifyResponse.ok) {
          const verifyData = await verifyResponse.json();
          const foundProperty = verifyData.properties?.find((p: any) => p.id === createResult.id);
          
          if (foundProperty) {
            console.log('✅ VERIFICATION SUCCESS! Property found in database');
            console.log('Found property:', foundProperty);
          } else {
            console.error('❌ VERIFICATION FAILED! Property not found in database');
            console.log('All properties:', verifyData.properties?.map((p: any) => ({ id: p.id, title: p.title })));
          }
        }
      } else {
        const createError = await createResponse.text();
        console.error('❌ Property creation failed');
        console.error('Error:', createError);
      }
      
    } catch (error) {
      console.error('❌ Debug session failed:', error);
    }
    
    console.log('🔧 === API DEBUG SESSION ENDED ===');
  },

  // Search properties by query
  searchProperties: async (query: string) => {
    try {
      const response = await fetch(`/api/properties/?search=${encodeURIComponent(query)}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error searching properties:', error);
      return [];
    }
  },

  // Fetch properties with filters
  fetchPropertiesWithFilters: async (filters: {
    search?: string;
    listing_type?: string;
    property_type?: string;
    min_price?: number;
    max_price?: number;
    city?: string;
    ordering?: string;
    bedrooms?: string;
    bathrooms?: string;
    furnishing?: string;
    monthly_rent?: number;
    loan_available?: boolean;
    amenities?: string;
  }) => {
    try {
      const params = new URLSearchParams();
      
      Object.entries(filters).forEach(([key, value]) => {
        if (value !== undefined && value !== null && value !== '') {
          params.append(key, value.toString());
        }
      });
      
      console.log('🔍 Enhanced search with filters:', filters);
      console.log('🔍 Search URL params:', params.toString());
      
      const response = await fetch(`/api/properties/?limit=100&${params.toString()}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching filtered properties:', error);
      return [];
    }
  },

  // Fetch properties with pagination
  fetchPropertiesPaginated: async (page: number = 1, pageSize: number = 12) => {
    try {
      const response = await fetch(`/api/properties/?page=${page}&page_size=${pageSize}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return {
          properties: result.properties || [],
          total: result.total || 0,
          page: result.page || 1,
          totalPages: result.total_pages || 1,
          hasNext: result.has_next || false,
          hasPrevious: result.has_previous || false
        };
      }
      return {
        properties: [],
        total: 0,
        page: 1,
        totalPages: 1,
        hasNext: false,
        hasPrevious: false
      };
    } catch (error) {
      console.error('Error fetching paginated properties:', error);
      return {
        properties: [],
        total: 0,
        page: 1,
        totalPages: 1,
        hasNext: false,
        hasPrevious: false
      };
    }
  },

  // Store search history
  storeSearchHistory: async (searchData: {
    user_id: string;
    query: string;
    filters: any;
    location: string;
  }) => {
    try {
      const response = await fetch('/api/search-history/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify(searchData),
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        return await response.json();
      }
      return null;
    } catch (error) {
      console.error('Error storing search history:', error);
      return null;
    }
  },

  // Get recommendations
  getRecommendations: async (userId: string) => {
    try {
      const response = await fetch(`/api/search-history/recommendations/${userId}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        return await response.json();
      }
      return [];
    } catch (error) {
      console.error('Error fetching recommendations:', error);
      return [];
    }
  },

  // Geocoding API functions - Nominatim primary, Geoapify fallback
  geocodeAddress: async (params: {
    landmark?: string;
    pincode?: string;
    city?: string;
    state?: string;
  }) => {
    // Build search query for Nominatim
    const searchParts = [];
    if (params.landmark) searchParts.push(params.landmark);
    if (params.city) searchParts.push(params.city);
    if (params.state) searchParts.push(params.state);
    if (params.pincode) searchParts.push(params.pincode);
    
    const searchQuery = searchParts.join(', ');
    
    if (!searchQuery.trim()) {
      return {
        success: false,
        error: 'No search parameters provided'
      };
    }

    try {
      console.log('🗺️ Trying Nominatim API first for:', searchQuery);
      
      // Try Nominatim API first (OpenStreetMap)
      const nominatimUrl = `https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(searchQuery)}&addressdetails=1&limit=1&countrycodes=in`;
      
      const nominatimResponse = await fetch(nominatimUrl, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
          'User-Agent': 'NAL-Property-App/1.0'
        }
      });
      
      if (nominatimResponse.ok) {
        const nominatimData = await nominatimResponse.json();
        
        if (nominatimData && nominatimData.length > 0) {
          const result = nominatimData[0];
          return {
            success: true,
            latitude: parseFloat(result.lat),
            longitude: parseFloat(result.lon),
            displayName: result.display_name,
            address: {
              city: result.address?.city || result.address?.town || result.address?.village,
              state: result.address?.state,
              postcode: result.address?.postcode,
              suburb: result.address?.suburb || result.address?.neighbourhood,
              road: result.address?.road
            }
          };
        }
      }
      
      console.log('⚠️ Nominatim failed, trying Geoapify fallback...');
      
      // Fallback to Geoapify API
      const GEOAPIFY_API_KEY = '9b3807944bbe44d2a0b29a5706c15bf9';
      const geoapifyUrl = `https://api.geoapify.com/v1/geocode/search?text=${encodeURIComponent(searchQuery)}&apiKey=${GEOAPIFY_API_KEY}&format=json&limit=1`;
      
      const geoapifyResponse = await fetch(geoapifyUrl, {
        method: 'GET',
        headers: {
          'Accept': 'application/json'
        }
      });
      
      if (geoapifyResponse.ok) {
        const geoapifyData = await geoapifyResponse.json();
        
        if (geoapifyData && geoapifyData.results && geoapifyData.results.length > 0) {
          const result = geoapifyData.results[0];
          return {
            success: true,
            latitude: result.lat,
            longitude: result.lon,
            displayName: result.formatted,
            address: {
              city: result.city,
              state: result.state,
              postcode: result.postcode,
              suburb: result.suburb || result.district,
              road: result.street
            }
          };
        }
      }
      
      return {
        success: false,
        error: 'Location not found in both Nominatim and Geoapify'
      };
      
    } catch (error) {
      console.error('Geocoding error:', error);
      return {
        success: false,
        error: 'Network error during geocoding'
      };
    }
  },

  reverseGeocode: async (latitude: number, longitude: number) => {
    try {
      console.log('🗺️ Trying Nominatim reverse geocoding first for:', latitude, longitude);
      
      // Try Nominatim API first (OpenStreetMap)
      const nominatimUrl = `https://nominatim.openstreetmap.org/reverse?format=json&lat=${latitude}&lon=${longitude}&addressdetails=1&accept-language=en`;
      
      const nominatimResponse = await fetch(nominatimUrl, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
          'User-Agent': 'NAL-Property-App/1.0'
        }
      });
      
      if (nominatimResponse.ok) {
        const nominatimData = await nominatimResponse.json();
        
        if (nominatimData && nominatimData.address) {
          return {
            success: true,
            displayName: nominatimData.display_name,
            address: {
              city: nominatimData.address.city || nominatimData.address.town || nominatimData.address.village,
              state: nominatimData.address.state,
              postcode: nominatimData.address.postcode,
              suburb: nominatimData.address.suburb || nominatimData.address.neighbourhood,
              road: nominatimData.address.road
            }
          };
        }
      }
      
      console.log('⚠️ Nominatim reverse geocoding failed, trying Geoapify fallback...');
      
      // Fallback to Geoapify API
      const GEOAPIFY_API_KEY = '9b3807944bbe44d2a0b29a5706c15bf9';
      const geoapifyUrl = `https://api.geoapify.com/v1/geocode/reverse?lat=${latitude}&lon=${longitude}&apiKey=${GEOAPIFY_API_KEY}&format=json`;
      
      const geoapifyResponse = await fetch(geoapifyUrl, {
        method: 'GET',
        headers: {
          'Accept': 'application/json'
        }
      });
      
      if (geoapifyResponse.ok) {
        const geoapifyData = await geoapifyResponse.json();
        
        if (geoapifyData && geoapifyData.results && geoapifyData.results.length > 0) {
          const result = geoapifyData.results[0];
          return {
            success: true,
            displayName: result.formatted,
            address: {
              city: result.city,
              state: result.state,
              postcode: result.postcode,
              suburb: result.suburb || result.district,
              road: result.street
            }
          };
        }
      }
      
      return {
        success: false,
        error: 'Address not found in both Nominatim and Geoapify'
      };
      
    } catch (error) {
      console.error('Reverse geocoding error:', error);
      return {
        success: false,
        error: 'Network error during reverse geocoding'
      };
    }
  },

  // Enhanced AI search suggestions
  getSearchSuggestions: async (query: string) => {
    try {
      const response = await fetch(`/api/search-history/suggestions?q=${encodeURIComponent(query)}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.suggestions || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching search suggestions:', error);
      return [];
    }
  },

  // Get smart property suggestions based on user behavior
  getSmartSuggestions: async (query: string, userPreferences?: any) => {
    try {
      const params = new URLSearchParams();
      params.append('q', query);
      if (userPreferences) {
        params.append('preferences', JSON.stringify(userPreferences));
      }
      
      const response = await fetch(`/api/smart-suggestions/?${params.toString()}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.suggestions || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching smart suggestions:', error);
      return [];
    }
  },

  // Get trending searches and popular queries
  getTrendingSearches: async () => {
    try {
      const response = await fetch('/api/trending-searches/', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.trending || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching trending searches:', error);
      return [];
    }
  },

  // Debug function specifically for lease submissions
  debugLeaseSubmission: async () => {
    console.log('🏢 === DEBUGGING LEASE SUBMISSION ===');
    
    const testLeaseData = {
      // Basic required fields
      title: 'Debug Lease Property',
      description: 'Test lease property for debugging',
      listing_type: 'lease',
      property_type: 'commercial',
      status: 'pending',
      address: '123 Lease Test Street',
      city: 'Bangalore',
      locality: 'Test Locality',
      state: 'Karnataka',
      pincode: '560001',
      latitude: 12.9716,
      longitude: 77.5946,
      built_up_area: 2000,
      bedrooms: 0, // Commercial property
      bathrooms: 2,
      facing: 'north',
      furnishing: 'unfurnished',
      property_age: '1-5',
      possession_status: 'ready-to-move',
      amenities: ['parking', 'security'],
      custom_amenities: [],
      property_highlights: ['commercial space'],
      nearby_places: ['metro station'],
      primary_image: null,
      virtual_tour_url: null,
      image_urls: [],
      listed_by_id: 'debug_lease_user',
      fullname: 'Debug Lease User',
      owner_name: 'Debug Lease User',
      mobile: '9999999999',
      email: 'debug.lease@example.com',
      property_manager_contact: '9999999999',
      contact_type: 'owner',
      
      // Lease-specific required fields
      lease_type: 'commercial',
      lease_duration: '3-years',
      monthly_lease_amount: 50000,
      lease_start_date: new Date('2024-02-01').toISOString().split('T')[0],
      possession_availability: 'immediate',
      
      // Optional lease fields
      advance_deposit: 200000,
      maintenance_charges: 5000,
      lock_in_period: '1-year',
      payment_frequency: 'monthly',
      maintenance_responsibility: 'tenant',
      auto_renew: false
    };
    
    console.log('📤 Test lease data:', JSON.stringify(testLeaseData, null, 2));
    
    try {
      const response = await fetch('/api/properties/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(testLeaseData)
      });
      
      console.log('📥 Response status:', response.status);
      console.log('📥 Response headers:', [...response.headers.entries()]);
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ Lease property created successfully!');
        console.log('✅ Response:', JSON.stringify(result, null, 2));
        return { success: true, result };
      } else {
        const errorText = await response.text();
        console.error('❌ Lease submission failed');
        console.error('❌ Status:', response.status);
        console.error('❌ Error response:', errorText);
        
        // Try to parse and display the error properly
        try {
          const errorJson = JSON.parse(errorText);
          console.error('❌ Parsed error:', errorJson);
          
          // Extract meaningful error message
          let errorMessage = 'Unknown error';
          if (errorJson.detail) {
            if (Array.isArray(errorJson.detail)) {
              errorMessage = errorJson.detail.map((item: any) => {
                if (typeof item === 'string') return item;
                if (item.msg) return item.msg;
                return JSON.stringify(item);
              }).join(', ');
            } else {
              errorMessage = String(errorJson.detail);
            }
          } else if (errorJson.message) {
            errorMessage = errorJson.message;
          }
          
          console.error('❌ Extracted error message:', errorMessage);
          return { success: false, error: errorMessage };
        } catch (parseError) {
          console.error('❌ Could not parse error JSON:', parseError);
          return { success: false, error: errorText };
        }
      }
    } catch (error) {
      console.error('❌ Network error:', error);
      return { success: false, error: error instanceof Error ? error.message : String(error) };
    }
  },

  // Debug function to check urgent sale and bidding properties
  debugUrgentAndBidding: async () => {
    console.log('🔍 === DEBUGGING URGENT SALE & BIDDING PROPERTIES ===');
    
    try {
      // Fetch all properties first
      const response = await fetch('/api/properties/', {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        const result = await response.json();
        const allProperties = result.properties || [];
        
        console.log('📊 Total properties in database:', allProperties.length);
        console.log('📊 Sample property structure:', allProperties[0]);
        
        // Analyze listing types
        const listingTypes: { [key: string]: number } = {};
        allProperties.forEach((prop: any) => {
          const type = prop.listing_type || 'undefined';
          listingTypes[type] = (listingTypes[type] || 0) + 1;
        });
        
        console.log('📊 Listing types breakdown:', listingTypes);
        
        // Check for urgent sale indicators
        const urgentSaleIndicators = allProperties.map((prop: any) => ({
          id: prop.id,
          title: prop.title,
          listing_type: prop.listing_type,
          urgent_sale: prop.urgent_sale,
          is_urgent_sale: prop.is_urgent_sale,
          urgent_sale_end_date: prop.urgent_sale_end_date,
          urgent_sale_end_time: prop.urgent_sale_end_time
        }));
        
        console.log('🚨 Urgent sale indicators:', urgentSaleIndicators);
        
        // Check for bidding indicators
        const biddingIndicators = allProperties.map((prop: any) => ({
          id: prop.id,
          title: prop.title,
          listing_type: prop.listing_type,
          is_auction: prop.is_auction,
          auction_enabled: prop.auction_enabled,
          starting_bid_price: prop.starting_bid_price,
          auction_end_date: prop.auction_end_date,
          auction_end_time: prop.auction_end_time
        }));
        
        console.log('🏆 Bidding indicators:', biddingIndicators);
        
        // Test the actual API functions
        console.log('\n🧪 Testing fetchUrgentSaleProperties...');
        const urgentSales = await api.fetchUrgentSaleProperties();
        console.log('✅ Urgent sale properties found:', urgentSales.length);
        console.log('📋 Urgent sale properties:', urgentSales);
        
        console.log('\n🧪 Testing fetchBiddingProperties...');
        const biddingProps = await api.fetchBiddingProperties();
        console.log('✅ Bidding properties found:', biddingProps.length);
        console.log('📋 Bidding properties:', biddingProps);
        
        return {
          totalProperties: allProperties.length,
          listingTypes,
          urgentSaleCount: urgentSales.length,
          biddingCount: biddingProps.length,
          urgentSales,
          biddingProps
        };
      } else {
        console.error('❌ Failed to fetch properties:', response.status);
        return null;
      }
    } catch (error) {
      console.error('❌ Debug failed:', error);
      return null;
    }
  },

  // Upload document with JWT authentication
  uploadDocument: async (file: File, documentType: string = 'general', propertyId?: string) => {
    try {
      const formData = new FormData();
      formData.append('file', file);
      
      const params = new URLSearchParams();
      params.append('document_type', documentType);
      if (propertyId) {
        params.append('property_id', propertyId);
      }
      
      const response = await fetch(`/api/documents/upload?${params.toString()}`, {
        method: 'POST',
        headers: {
          'Authorization': 'Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJ1c2VyX2lkIjoiMjc2MTc2MmItMjIzMy00ZTc2LTk1MTYtZGUyZDAzNDM5NTk0IiwicGhvbmVfbnVtYmVyIjoiIiwicm9sZV9pZCI6MSwiaXNfYWRtaW4iOmZhbHNlLCJ0eXBlIjoiYWNjZXNzIiwiZXhwIjoxNzYxMDQwMTkyfQ.5JNYQRZMGzVs3ewgREIJ44PwovWR_JSoH-Yu51YkJOM'
        },
        body: formData
      });
      
      if (response.ok) {
        const result = await response.json();
        return {
          success: true,
          document_id: result.document_id,
          filename: result.filename,
          file_size: result.file_size,
          document_type: result.document_type,
          uploaded_at: result.uploaded_at
        };
      } else {
        const errorData = await response.json().catch(() => ({ message: 'Upload failed' }));
        return {
          success: false,
          error: errorData.message || `Upload failed with status ${response.status}`
        };
      }
    } catch (error) {
      console.error('Document upload error:', error);
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Network error during upload'
      };
    }
  },

  // Test JWT token validity
  testJWT: async () => {
    try {
      const response = await fetch('/api/documents/upload?document_type=general&property_id=1', {
        method: 'POST',
        headers: {
          'Accept': 'application/json',
          'Authorization': 'Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJ1c2VyX2lkIjoiMjAwZDg3MzEtNzNmNS00NDM0LWI4MzQtYzFiMTJmMWE4Yjc0IiwicGhvbmVfbnVtYmVyIjoiIiwicm9sZV9pZCI6MSwiaXNfYWRtaW4iOmZhbHNlLCJ0eXBlIjoiYWNjZXNzIiwiZXhwIjoxNzYxMDM4MjYzfQ.XBGvF_JXvPrZOQCD4BhrexngWqlnyt2B2pxA9_Wow7w'
        },
        body: new FormData()
      });
      
      console.log('JWT Test Response:', response.status, response.statusText);
      const result = await response.text();
      console.log('JWT Test Body:', result);
      
      return { status: response.status, body: result };
    } catch (error) {
      console.error('JWT Test Error:', error);
      return { error: error instanceof Error ? error.message : 'Test failed' };
    }
  },

  // Smart Suggestions API methods
  fetchTrendingProperties: async (city?: string) => {
    try {
      const url = city ? `/api/properties/trending?city=${encodeURIComponent(city)}` : '/api/properties/trending';
      const response = await fetch(url, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching trending properties:', error);
      return [];
    }
  },

  fetchNearbyProperties: async (lat: number, lng: number) => {
    try {
      const response = await fetch(`/api/properties/nearby?lat=${lat}&lng=${lng}`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching nearby properties:', error);
      return [];
    }
  },

  // Fetch properties for seller dashboard
  fetchProperties: async () => {
    try {
      console.log('🔍 Fetching properties for seller dashboard...');
      const response = await fetch('/api/properties/', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching properties:', error);
      return [];
    }
  },

  // Clean up duplicate properties
  cleanupDuplicates: async () => {
    try {
      console.log('🧹 Cleaning up duplicate properties...');
      // This is a placeholder - in a real app, this would call a backend endpoint
      return { deleted_ids: [] };
    } catch (error) {
      console.error('Error cleaning up duplicates:', error);
      return { deleted_ids: [] };
    }
  },

  // Delete property
  deleteProperty: async (propertyId: string) => {
    try {
      console.log('🗑️ Deleting property:', propertyId);
      const response = await fetch(`/api/properties/${propertyId}`, {
        method: 'DELETE',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        console.log('✅ Property deleted successfully');
        return { success: true };
      } else {
        const errorText = await response.text();
        throw new Error(`Delete failed: ${errorText}`);
      }
    } catch (error) {
      console.error('Error deleting property:', error);
      throw error;
    }
  },

  // List asset for sale
  listAssetForSale: async (propertyId: string) => {
    try {
      console.log('🚀 Listing asset for sale:', propertyId);
      
      // First fetch the property details
      const property = await api.fetchPropertyById(propertyId);
      if (!property) {
        throw new Error('Property not found');
      }
      
      // Return the property details as seller details
      const sellerDetails = {
        fullname: property.fullname || property.owner_name || '',
        email: property.email || '',
        mobile: property.mobile || property.property_manager_contact || '',
        bio: property.bio || '',
        ownership_type: property.ownership_type || '',
        selling_method: property.selling_method || '',
        selling_timeline: property.selling_timeline || '',
        price_flexibility: property.price_flexibility || false
      };
      
      console.log('✅ Asset listing prepared with seller details:', sellerDetails);
      
      return {
        success: true,
        message: 'Asset ready for listing',
        seller_details: sellerDetails,
        property: property
      };
    } catch (error) {
      console.error('❌ Error listing asset for sale:', error);
      throw error;
    }
  },

  // Fetch properties by seller/user ID
  fetchPropertiesBySellerId: async (sellerId: string) => {
    try {
      console.log('🔍 Fetching properties for seller ID:', sellerId);
      const response = await fetch(`/api/properties/?listed_by_id=${sellerId}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
      });
      
      if (response.ok) {
        const result = await response.json();
        console.log('📊 Properties fetched for seller:', result.properties?.length || 0);
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error fetching properties by seller ID:', error);
      return [];
    }
  },

  // Advanced search with AI-powered filtering
  advancedSearch: async (searchParams: {
    query?: string;
    bhk?: string;
    location?: string;
    priceRange?: { min?: number; max?: number };
    propertyType?: string;
    status?: string;
    amenities?: string[];
  }) => {
    try {
      const params = new URLSearchParams();
      
      if (searchParams.query) params.append('q', searchParams.query);
      if (searchParams.bhk) params.append('bedrooms', searchParams.bhk);
      if (searchParams.location) params.append('location', searchParams.location);
      if (searchParams.propertyType) params.append('property_type', searchParams.propertyType);
      if (searchParams.status) params.append('status', searchParams.status);
      if (searchParams.priceRange?.min) params.append('min_price', searchParams.priceRange.min.toString());
      if (searchParams.priceRange?.max) params.append('max_price', searchParams.priceRange.max.toString());
      if (searchParams.amenities?.length) params.append('amenities', searchParams.amenities.join(','));
      
      const response = await fetch(`/api/properties/advanced-search?${params.toString()}`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        const result = await response.json();
        return result.properties || [];
      }
      return [];
    } catch (error) {
      console.error('Error in advanced search:', error);
      return [];
    }
  }
};

// Make debug functions globally available
if (typeof window !== 'undefined') {
  (window as any).debugAPI = api.debugAPI;
  (window as any).debugUrgentAndBidding = api.debugUrgentAndBidding;
  (window as any).debugLeaseSubmission = api.debugLeaseSubmission;
}