// Enhanced API Error Handler and Fallback Utility
const API_BASE_URL = import.meta.env.VITE_API_BASE?.replace('/api', '') || 'http://localhost:8090';

export interface APIError {
  message: string;
  status?: number;
  isNetworkError: boolean;
  isServerError: boolean;
  isClientError: boolean;
  isAuthError: boolean;
}

export const createAPIError = (error: any, status?: number): APIError => {
  const message = error instanceof Error ? error.message : String(error);
  
  return {
    message,
    status,
    isNetworkError: !status || status === 0,
    isServerError: status ? status >= 500 : false,
    isClientError: status ? status >= 400 && status < 500 : false,
    isAuthError: status ? status === 401 || status === 403 : false
  };
};

export const handleAPIError = (error: APIError): void => {
  console.error('🚨 API Error Handler:', {
    message: error.message,
    status: error.status,
    type: error.isNetworkError ? 'Network' : 
          error.isServerError ? 'Server' : 
          error.isClientError ? 'Client' : 
          error.isAuthError ? 'Authentication' : 'Unknown'
  });

  // Store error for debugging
  const errorLog = JSON.parse(localStorage.getItem('apiErrorLog') || '[]');
  errorLog.push({
    timestamp: new Date().toISOString(),
    error: error.message,
    status: error.status,
    url: window.location.pathname
  });
  
  // Keep only last 10 errors
  if (errorLog.length > 10) {
    errorLog.shift();
  }
  
  localStorage.setItem('apiErrorLog', JSON.stringify(errorLog));
};

export const safeAPICall = async <T>(
  apiCall: () => Promise<T>,
  fallbackData?: T,
  customErrorHandler?: (error: APIError) => void
): Promise<T> => {
  try {
    return await apiCall();
  } catch (error) {
    const apiError = createAPIError(error, (error as any)?.status);
    
    // Handle the error
    if (customErrorHandler) {
      customErrorHandler(apiError);
    } else {
      handleAPIError(apiError);
    }
    
    // Return fallback data if available
    if (fallbackData !== undefined) {
      console.log('🔄 Using fallback data due to API error');
      return fallbackData;
    }
    
    // Re-throw if no fallback available
    throw error;
  }
};

export const createMockPendingProperties = () => ({
  properties: [],
  total_count: 0,
  total_pages: 0,
  current_page: 1,
  page_size: 20
});

// Handle specific backend validation errors
export const handleValidationError = (error: string): boolean => {
  return error.includes('contact_type') && error.includes('Input should be a valid string');
};

export const createValidationErrorResponse = () => ({
  error: 'Backend data validation issue. Some properties have missing contact_type. Please run database fix.',
  suggestion: 'Run: UPDATE nal.properties SET contact_type = \'owner\' WHERE contact_type IS NULL;'
});

export const createMockNotifications = () => ({
  notifications: [],
  total_count: 0,
  unread_count: 0
});

export const checkBackendStatus = async (): Promise<{
  isOnline: boolean;
  hasDatabase: boolean;
  responseTime: number;
  error?: string;
}> => {
  const startTime = Date.now();
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/properties/`, {
      method: 'GET',
      headers: { 'Accept': 'application/json' },
      signal: AbortSignal.timeout(3000)
    });
    
    const responseTime = Date.now() - startTime;
    
    if (response.ok) {
      const data = await response.json();
      return {
        isOnline: true,
        hasDatabase: Array.isArray(data.properties),
        responseTime
      };
    } else {
      return {
        isOnline: false,
        hasDatabase: false,
        responseTime,
        error: `HTTP ${response.status}`
      };
    }
  } catch (error) {
    return {
      isOnline: false,
      hasDatabase: false,
      responseTime: Date.now() - startTime,
      error: error instanceof Error ? error.message : 'Unknown error'
    };
  }
};

// Enhanced fetch wrapper with retry logic
export const enhancedFetch = async (
  url: string,
  options: RequestInit = {},
  maxRetries: number = 2
): Promise<Response> => {
  let lastError: any;
  
  for (let attempt = 1; attempt <= maxRetries + 1; attempt++) {
    try {
      console.log(`🔄 API Attempt ${attempt}/${maxRetries + 1}:`, url);
      
      const response = await fetch(url, {
        ...options,
        signal: AbortSignal.timeout(5000)
      });
      
      if (response.ok) {
        console.log(`✅ API Success on attempt ${attempt}:`, url);
        return response;
      }
      
      // Don't retry on client errors (4xx)
      if (response.status >= 400 && response.status < 500) {
        console.log(`🚫 Client error ${response.status}, not retrying:`, url);
        return response;
      }
      
      // Store error for potential retry
      lastError = new Error(`HTTP ${response.status}: ${response.statusText}`);
      
      if (attempt <= maxRetries) {
        console.log(`⏳ Waiting before retry ${attempt + 1}...`);
        await new Promise(resolve => setTimeout(resolve, 1000 * attempt));
      }
    } catch (error) {
      lastError = error;
      
      if (attempt <= maxRetries) {
        console.log(`⏳ Network error, retrying ${attempt + 1}...`);
        await new Promise(resolve => setTimeout(resolve, 1000 * attempt));
      }
    }
  }
  
  console.log(`❌ All ${maxRetries + 1} attempts failed for:`, url);
  throw lastError;
};

// Make debugging functions globally available
if (typeof window !== 'undefined') {
  (window as any).checkBackendStatus = checkBackendStatus;
  (window as any).getAPIErrorLog = () => JSON.parse(localStorage.getItem('apiErrorLog') || '[]');
  (window as any).clearAPIErrorLog = () => localStorage.removeItem('apiErrorLog');
}

export default {
  createAPIError,
  handleAPIError,
  safeAPICall,
  createMockPendingProperties,
  createMockNotifications,
  checkBackendStatus,
  enhancedFetch
};
