// API Health Check Utility
// This utility helps diagnose and handle API connectivity issues

const API_BASE_URL = import.meta.env.VITE_API_BASE?.replace('/api', '') || 'http://localhost:8090';

export interface HealthCheckResult {
  isHealthy: boolean;
  endpoints: {
    [key: string]: {
      status: number;
      ok: boolean;
      error?: string;
      responseTime?: number;
    };
  };
  overallStatus: 'healthy' | 'degraded' | 'unhealthy';
  summary: string;
}

export interface ApiEndpoint {
  name: string;
  path: string;
  method: 'GET' | 'POST';
  requiresAuth?: boolean;
  critical?: boolean;
}

// List of critical API endpoints to check
const CRITICAL_ENDPOINTS: ApiEndpoint[] = [
  {
    name: 'properties',
    path: '/api/properties/',
    method: 'GET',
    critical: true
  },
  {
    name: 'admin_properties_pending',
    path: '/api/admin/properties/pending',
    method: 'GET',
    requiresAuth: true,
    critical: true
  },
  {
    name: 'admin_notifications',
    path: '/api/admin/notifications',
    method: 'GET',
    requiresAuth: true,
    critical: false
  },
  {
    name: 'admin_visit_schedules',
    path: '/api/admin/visit-schedules',
    method: 'GET',
    requiresAuth: true,
    critical: false
  }
];

// Check individual endpoint health
const checkEndpoint = async (endpoint: ApiEndpoint): Promise<{
  status: number;
  ok: boolean;
  error?: string;
  responseTime?: number;
}> => {
  const startTime = Date.now();
  const url = `${API_BASE_URL}${endpoint.path}`;
  
  try {
    console.log(`🔍 Checking endpoint health: ${endpoint.name} (${url})`);
    
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      'Accept': 'application/json'
    };

    // Add auth token if required
    if (endpoint.requiresAuth) {
      const token = localStorage.getItem('access_token');
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
    }

    const response = await fetch(url, {
      method: endpoint.method,
      headers,
      signal: AbortSignal.timeout(5000) // 5 second timeout
    });

    const responseTime = Date.now() - startTime;
    
    console.log(`📥 Endpoint ${endpoint.name} response:`, {
      status: response.status,
      ok: response.ok,
      responseTime: `${responseTime}ms`
    });

    if (response.ok) {
      return {
        status: response.status,
        ok: true,
        responseTime
      };
    } else {
      let errorDetails = `HTTP ${response.status}`;
      try {
        const errorText = await response.text();
        if (errorText) {
          errorDetails += ` - ${errorText.substring(0, 100)}`;
        }
      } catch {
        // Ignore error parsing failures
      }
      
      return {
        status: response.status,
        ok: false,
        error: errorDetails,
        responseTime
      };
    }
  } catch (error) {
    const responseTime = Date.now() - startTime;
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    
    console.error(`❌ Endpoint ${endpoint.name} failed:`, errorMessage);
    
    return {
      status: 0,
      ok: false,
      error: errorMessage,
      responseTime
    };
  }
};

// Perform comprehensive health check
export const performHealthCheck = async (): Promise<HealthCheckResult> => {
  console.log('🏥 Starting API health check...');
  
  const results: HealthCheckResult['endpoints'] = {};
  let healthyCount = 0;
  let criticalFailures = 0;
  
  // Check all endpoints in parallel
  const endpointChecks = CRITICAL_ENDPOINTS.map(async (endpoint) => {
    const result = await checkEndpoint(endpoint);
    return { endpoint, result };
  });
  
  const completedChecks = await Promise.all(endpointChecks);
  
  // Process results
  completedChecks.forEach(({ endpoint, result }) => {
    results[endpoint.name] = result;
    
    if (result.ok) {
      healthyCount++;
    } else if (endpoint.critical) {
      criticalFailures++;
    }
  });
  
  // Determine overall status
  let overallStatus: HealthCheckResult['overallStatus'];
  let summary: string;
  
  if (criticalFailures > 0) {
    overallStatus = 'unhealthy';
    summary = `Critical failures detected: ${criticalFailures} critical endpoint(s) failing`;
  } else if (healthyCount === CRITICAL_ENDPOINTS.length) {
    overallStatus = 'healthy';
    summary = 'All endpoints are responding normally';
  } else {
    overallStatus = 'degraded';
    summary = `${CRITICAL_ENDPOINTS.length - healthyCount} endpoint(s) experiencing issues`;
  }
  
  const healthResult: HealthCheckResult = {
    isHealthy: overallStatus === 'healthy',
    endpoints: results,
    overallStatus,
    summary
  };
  
  console.log('🏥 Health check completed:', healthResult);
  return healthResult;
};

// Quick health check for critical endpoints only
export const quickHealthCheck = async (): Promise<boolean> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/properties/`, {
      method: 'GET',
      headers: { 'Accept': 'application/json' },
      signal: AbortSignal.timeout(3000)
    });
    return response.ok;
  } catch (error) {
    console.warn('Quick health check failed:', error);
    return false;
  }
};

// Get fallback data when API is unavailable
export const getFallbackData = async (dataType: 'pendingProperties' | 'notifications') => {
  console.log(`🔄 Providing fallback data for: ${dataType}`);
  
  switch (dataType) {
    case 'pendingProperties':
      return {
        properties: [],
        total_count: 0,
        total_pages: 0,
        current_page: 1,
        page_size: 20
      };
      
    case 'notifications':
      return {
        notifications: [],
        total_count: 0,
        unread_count: 0
      };
      
    default:
      return null;
  }
};

// Enhanced fetch with retry logic
export const enhancedFetch = async (
  url: string,
  options: RequestInit = {},
  retries: number = 2
): Promise<Response> => {
  const lastError = new Error();
  
  for (let attempt = 0; attempt <= retries; attempt++) {
    try {
      console.log(`🔄 Fetch attempt ${attempt + 1}/${retries + 1} for: ${url}`);
      
      const response = await fetch(url, {
        ...options,
        signal: AbortSignal.timeout(5000)
      });
      
      if (response.ok) {
        console.log(`✅ Fetch successful on attempt ${attempt + 1}`);
        return response;
      }
      
      // If it's a 500 error, retry
      if (response.status === 500 && attempt < retries) {
        console.warn(`⚠️ Got 500 error, retrying... (${attempt + 1}/${retries})`);
        await new Promise(resolve => setTimeout(resolve, 1000 * (attempt + 1)));
        continue;
      }
      
      // For other errors, don't retry
      return response;
    } catch (error) {
      console.error(`❌ Fetch attempt ${attempt + 1} failed:`, error);
      lastError.message = error instanceof Error ? error.message : 'Unknown error';
      
      if (attempt < retries) {
        console.log(`🔄 Retrying in ${1000 * (attempt + 1)}ms...`);
        await new Promise(resolve => setTimeout(resolve, 1000 * (attempt + 1)));
      }
    }
  }
  
  throw lastError;
};

// Make health check available globally for debugging
if (typeof window !== 'undefined') {
  (window as any).apiHealthCheck = performHealthCheck;
  (window as any).quickHealthCheck = quickHealthCheck;
}

export default {
  performHealthCheck,
  quickHealthCheck,
  getFallbackData,
  enhancedFetch,
  checkEndpoint
};
