// User activity tracking
let lastActivity = Date.now();
let refreshTimer: NodeJS.Timeout | null = null;
let sessionTimer: NodeJS.Timeout | null = null;

// Track user activity
const trackActivity = () => {
  lastActivity = Date.now();
};

// Check if user is active (within last 5 minutes)
const isUserActive = (): boolean => {
  const fiveMinutes = 5 * 60 * 1000;
  return (Date.now() - lastActivity) < fiveMinutes;
};

// Auto logout after inactivity
const autoLogout = () => {
  secureLogout();
};

// Check session timeout (30 minutes of inactivity)
const checkSessionTimeout = (): boolean => {
  const thirtyMinutes = 30 * 60 * 1000;
  return (Date.now() - lastActivity) > thirtyMinutes;
};

// Token refresh utility with rotation
export const refreshToken = async (): Promise<boolean> => {
  const refreshToken = localStorage.getItem('refresh_token');
  if (!refreshToken) return false;

  try {
    const response = await fetch('/api/auth/refresh', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ refresh_token: refreshToken }),
    });

    if (response.ok) {
      const data = await response.json();
      // Store new tokens (refresh token rotation)
      localStorage.setItem('access_token', data.access_token);
      localStorage.setItem('refresh_token', data.refresh_token);
      return true;
    } else {
      // Refresh token expired, clear tokens
      localStorage.removeItem('access_token');
      localStorage.removeItem('refresh_token');
      return false;
    }
  } catch (err) {
    return false;
  }
};

// Enhanced fetch with automatic token refresh and role validation
export const authenticatedFetch = async (url: string, options: RequestInit = {}, requiredRole?: string): Promise<Response> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  // Validate token expiry
  try {
    const payload = JSON.parse(atob(accessToken.split('.')[1]));
    if (payload.exp && payload.exp < Date.now() / 1000) {
      localStorage.removeItem('access_token');
      throw new Error('Token expired');
    }

    // Check role requirement
    if (requiredRole && !payload.roles?.includes(requiredRole)) {
      throw new Error(`Role '${requiredRole}' required`);
    }
  } catch (e) {
    if (e instanceof Error && e.message.includes('Role')) {
      throw e;
    }
    // Token parsing failed, try refresh
  }

  // Add authorization header
  const headers = {
    ...options.headers,
    'Authorization': `Bearer ${accessToken}`,
    'Content-Type': 'application/json'
  };

  let response = await fetch(url, { ...options, headers });

  // If 401 or 403, try to refresh token and retry
  if (response.status === 401) {
    const refreshed = await refreshToken();
    if (refreshed) {
      const newAccessToken = localStorage.getItem('access_token');
      const newHeaders = {
        ...options.headers,
        'Authorization': `Bearer ${newAccessToken}`,
        'Content-Type': 'application/json'
      };
      response = await fetch(url, { ...options, headers: newHeaders });
    } else {
      // Refresh failed, redirect based on user type
      const isAgentUser = localStorage.getItem('agentToken') || localStorage.getItem('agentEmail');
      if (isAgentUser) {
        window.location.href = '/';
      } else {
        window.location.href = '/login';
      }
    }
  }

  if (response.status === 403) {
    throw new Error('Insufficient permissions');
  }

  return response;
};

// Check if token needs refresh (refresh 5 minutes before expiry)
export const checkTokenExpiry = (): boolean => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) return false;

  try {
    const payload = JSON.parse(atob(accessToken.split('.')[1]));
    const expiryTime = payload.exp * 1000; // Convert to milliseconds
    const currentTime = Date.now();
    const fiveMinutes = 5 * 60 * 1000; // 5 minutes in milliseconds

    return (expiryTime - currentTime) < fiveMinutes;
  } catch {
    return false;
  }
};

// Auto refresh token if needed and user is active
export const autoRefreshToken = async (): Promise<void> => {
  if (checkTokenExpiry() && isUserActive()) {
    await refreshToken();
  }
};

// Initialize activity tracking with security checks
export const initializeActivityTracking = () => {
  console.log('🔄 Initializing activity tracking...');

  // Check if we have any token before proceeding
  const accessToken = localStorage.getItem('access_token');
  const agentToken = localStorage.getItem('agentToken');
  const sellerToken = localStorage.getItem('sellerToken');

  if (!accessToken && !agentToken && !sellerToken) {
    console.log('❌ Activity tracking: No tokens found');
    return;
  }

  // Only validate session if we have access_token (not for agent/seller tokens)
  if (accessToken && !validateSession()) {
    console.log('❌ Activity tracking: Session validation failed');
    return;
  }

  console.log('✅ Activity tracking: Session validation passed');

  // Track user interactions
  const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart', 'click'];
  events.forEach(event => {
    document.addEventListener(event, trackActivity, true);
  });

  // Start automatic token refresh timer
  if (refreshTimer) clearInterval(refreshTimer);
  refreshTimer = setInterval(() => {
    // Only validate session for access_token users
    if (accessToken && !validateSession()) {
      cleanupActivityTracking();
      return;
    }

    if (isUserActive() && checkTokenExpiry()) {
      autoRefreshToken();
    }
  }, 30000); // Check every 30 seconds

  // Start session timeout timer
  if (sessionTimer) clearInterval(sessionTimer);
  sessionTimer = setInterval(() => {
    // Only validate session for access_token users
    if (accessToken && !validateSession()) {
      cleanupActivityTracking();
      return;
    }

    if (checkSessionTimeout()) {
      secureLogout();
    }
  }, 60000); // Check every minute
};

// Cleanup activity tracking
export const cleanupActivityTracking = () => {
  const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart', 'click'];
  events.forEach(event => {
    document.removeEventListener(event, trackActivity, true);
  });

  if (refreshTimer) {
    clearInterval(refreshTimer);
    refreshTimer = null;
  }

  if (sessionTimer) {
    clearInterval(sessionTimer);
    sessionTimer = null;
  }
};

export const isAdmin = (): boolean => {
  try {
    const accessToken = localStorage.getItem('access_token');
    if (accessToken) {
      const payload = JSON.parse(atob(accessToken.split('.')[1]));
      // Check if token is expired
      if (payload.exp && payload.exp < Date.now() / 1000) {
        localStorage.removeItem('access_token');
        return false;
      }
      // Check is_admin boolean, roles array, OR role_id (standardize on role_id 2 for admin)
      return (
        payload.is_admin === true ||
        payload.roles?.includes('Admin') ||
        payload.role_id === 2 ||
        false
      );
    }

    // Fallback check (less secure)
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    return (
      user.is_admin === true ||
      user.roles?.includes('Admin') ||
      user.role_id === 2 ||
      false
    );
  } catch {
    return false;
  }
};

export const hasRole = (role: string): boolean => {
  try {
    const accessToken = localStorage.getItem('access_token');
    if (accessToken) {
      const payload = JSON.parse(atob(accessToken.split('.')[1]));
      if (payload.exp && payload.exp < Date.now() / 1000) {
        localStorage.removeItem('access_token');
        return false;
      }
      return payload.roles?.includes(role) || false;
    }

    const user = JSON.parse(localStorage.getItem('user') || '{}');
    return user.roles?.includes(role) || false;
  } catch {
    return false;
  }
};

export const hasAnyRole = (roles: string[]): boolean => {
  try {
    const accessToken = localStorage.getItem('access_token');
    if (accessToken) {
      const payload = JSON.parse(atob(accessToken.split('.')[1]));
      if (payload.exp && payload.exp < Date.now() / 1000) {
        localStorage.removeItem('access_token');
        return false;
      }
      const userRoles = payload.roles || [];
      return roles.some(role => userRoles.includes(role));
    }

    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const userRoles = user.roles || [];
    return roles.some(role => userRoles.includes(role));
  } catch {
    return false;
  }
};

export const isSeller = (): boolean => hasRole('Seller');
export const isAgent = (): boolean => hasRole('Agent');
export const canManageProperties = (): boolean => hasAnyRole(['Admin', 'Seller', 'Agent']);
export const canApproveContent = (): boolean => hasRole('Admin');

export const isAuthenticated = (): boolean => {
  try {
    const accessToken = localStorage.getItem('access_token');
    if (!accessToken) return false;

    const payload = JSON.parse(atob(accessToken.split('.')[1]));
    const currentTime = Date.now() / 1000;

    if (payload.exp <= currentTime) {
      // Token expired, clean up
      localStorage.removeItem('access_token');
      localStorage.removeItem('refresh_token');
      localStorage.removeItem('user');
      return false;
    }

    return true;
  } catch {
    // Invalid token, clean up
    localStorage.removeItem('access_token');
    localStorage.removeItem('refresh_token');
    localStorage.removeItem('user');
    return false;
  }
};

// Secure logout function
export const secureLogout = (): void => {
  // Check if user is agent before clearing data
  const isAgentUser = localStorage.getItem('agentToken') || localStorage.getItem('agentEmail');

  // Clear all auth-related data
  localStorage.removeItem('access_token');
  localStorage.removeItem('refresh_token');
  localStorage.removeItem('user');
  localStorage.removeItem('adminLoggedIn');
  localStorage.removeItem('sellerToken');
  localStorage.removeItem('agentToken');

  // Clear any cached data
  sessionStorage.clear();

  // Redirect based on user type
  if (isAgentUser) {
    window.location.href = '/';
  } else {
    window.location.href = '/login';
  }
};

// Check for suspicious activity
export const validateSession = (): boolean => {
  const token = localStorage.getItem('access_token');
  if (!token) {
    console.log('🔍 validateSession: No token found');
    return false;
  }

  try {
    const payload = JSON.parse(atob(token.split('.')[1]));

    console.log('🔍 validateSession payload:', {
      user_id: payload.user_id,
      id: payload.id,
      sub: payload.sub,
      exp: payload.exp,
      currentTime: Date.now() / 1000,
      isExpired: payload.exp < Date.now() / 1000
    });

    // Only check for expiration - don't validate user_id structure as it varies by role
    if (!payload.exp) {
      console.log('❌ validateSession: No expiration in token');
      secureLogout();
      return false;
    }

    // Check for expired token
    if (payload.exp < Date.now() / 1000) {
      console.log('❌ validateSession: Token expired');
      secureLogout();
      return false;
    }

    console.log('✅ validateSession: Session valid');
    return true;
  } catch (e) {
    console.log('❌ validateSession: Error parsing token:', e);
    secureLogout();
    return false;
  }
};

export const getUserRoles = (): string[] => {
  try {
    const accessToken = localStorage.getItem('access_token');
    if (accessToken) {
      const payload = JSON.parse(atob(accessToken.split('.')[1]));
      if (payload.exp && payload.exp < Date.now() / 1000) {
        localStorage.removeItem('access_token');
        return [];
      }
      return payload.roles || [];
    }

    const user = JSON.parse(localStorage.getItem('user') || '{}');
    return user.roles || [];
  } catch {
    return [];
  }
};

export const getPrimaryRole = (): string => {
  const roles = getUserRoles();
  if (roles.includes('Admin')) return 'Admin';
  if (roles.includes('Agent')) return 'Agent';
  if (roles.includes('Seller')) return 'Seller';
  return 'User';
};

export const getUserRole = (): string => {
  return getPrimaryRole();
};

export const getPermissionLevel = (): number => {
  const roles = getUserRoles();
  if (roles.includes('Admin')) return 10;
  if (roles.includes('Agent') || roles.includes('Seller')) return 2;
  return 1;
};