// Behavior tracking utility for customer behavior analysis

export interface BehaviorEvent {
  user_id?: string;
  property_id?: number;
  session_id: string;
  event_type: EventType;
  event_data?: Record<string, any>;
  page_url?: string;
  referrer_url?: string;
  device_type?: DeviceType;
  browser_name?: string;
}

export enum EventType {
  PAGE_VIEW = 'page_view',
  PROPERTY_VIEW = 'property_view',
  PROPERTY_CLICK = 'property_click',
  SEARCH = 'search',
  FILTER_APPLY = 'filter_apply',
  SORT_CHANGE = 'sort_change',
  IMAGE_CLICK = 'image_click',
  DOCUMENT_CLICK = 'document_click',
  CONTACT_CLICK = 'contact_click',
  SAVE_PROPERTY = 'save_property',
  WISHLIST_ADD = 'wishlist_add',
  SCHEDULE_VISIT = 'schedule_visit',
  BID_PLACE = 'bid_place',
  AUCTION_REGISTER = 'auction_register',
  SCROLL = 'scroll',
  TIME_SPENT = 'time_spent',
  EXIT_INTENT = 'exit_intent'
}

export enum DeviceType {
  DESKTOP = 'desktop',
  MOBILE = 'mobile',
  TABLET = 'tablet'
}

class BehaviorTracker {
  private sessionId: string;
  private apiBaseUrl: string;
  private isEnabled: boolean;
  private pageStartTime: number;
  private scrollDepth: number = 0;
  private eventQueue: any[] = [];
  private isProcessing: boolean = false;

  constructor() {
    this.sessionId = this.getOrCreateSessionId();
    this.apiBaseUrl = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
    this.isEnabled = true;
    this.pageStartTime = Date.now();
    this.initializeTracking();
  }

  private getOrCreateSessionId(): string {
    // Use persistent session across tabs and page reloads
    let sessionId = localStorage.getItem('behavior_session_id');
    const sessionTimestamp = localStorage.getItem('behavior_session_timestamp');
    
    // Check if session expired (24 hours)
    if (sessionId && sessionTimestamp) {
      const sessionAge = Date.now() - parseInt(sessionTimestamp);
      if (sessionAge > 24 * 60 * 60 * 1000) {
        sessionId = null;
      }
    }
    
    if (!sessionId) {
      sessionId = crypto.randomUUID();
      localStorage.setItem('behavior_session_id', sessionId);
      localStorage.setItem('behavior_session_timestamp', Date.now().toString());
    }
    
    return sessionId;
  }

  private getDeviceType(): DeviceType {
    const width = window.innerWidth;
    if (width <= 768) return DeviceType.MOBILE;
    if (width <= 1024) return DeviceType.TABLET;
    return DeviceType.DESKTOP;
  }

  private getBrowserName(): string {
    const userAgent = navigator.userAgent;
    if (userAgent.includes('Chrome')) return 'Chrome';
    if (userAgent.includes('Firefox')) return 'Firefox';
    if (userAgent.includes('Safari')) return 'Safari';
    if (userAgent.includes('Edge')) return 'Edge';
    return 'Unknown';
  }

  private initializeTracking(): void {
    this.trackEvent(EventType.PAGE_VIEW, {
      url: window.location.href,
      title: document.title
    });

    let maxScroll = 0;
    window.addEventListener('scroll', () => {
      const scrollPercent = Math.round(
        (window.scrollY / (document.body.scrollHeight - window.innerHeight)) * 100
      );
      if (scrollPercent > maxScroll) {
        maxScroll = scrollPercent;
        this.scrollDepth = scrollPercent;
      }
    });

    window.addEventListener('beforeunload', () => {
      const timeSpent = Date.now() - this.pageStartTime;
      this.trackEvent(EventType.TIME_SPENT, {
        duration_ms: timeSpent,
        scroll_depth: this.scrollDepth
      });
    });

    document.addEventListener('mouseleave', (e) => {
      if (e.clientY <= 0) {
        this.trackEvent(EventType.EXIT_INTENT, {
          time_on_page: Date.now() - this.pageStartTime,
          scroll_depth: this.scrollDepth
        });
      }
    });
  }

  public async trackEvent(eventType: EventType, eventData?: Record<string, any>, propertyId?: number): Promise<void> {
    if (!this.isEnabled) return;

    try {
      console.log('📊 trackEvent called:', { eventType, eventData, propertyId });
      
      const sanitizedData = this.sanitizeEventData(eventData || {});
      console.log('📊 Sanitized event data:', sanitizedData);
      
      const event: BehaviorEvent = {
        session_id: this.sessionId,
        event_type: eventType,
        event_data: sanitizedData,
        page_url: window.location.href,
        referrer_url: document.referrer,
        device_type: this.getDeviceType(),
        browser_name: this.getBrowserName(),
        property_id: propertyId
      };

      console.log('📊 Final event object:', event);

      // Add to queue for batch processing
      this.eventQueue.push(event);
      
      // Process queue if not already processing
      if (!this.isProcessing) {
        this.processEventQueue();
      }
      
    } catch (error) {
      console.warn('❌ Tracking error:', error);
    }
  }

  private sanitizeEventData(data: Record<string, any>): Record<string, any> {
    console.log('🧼 Sanitizing event data:', data);
    const sanitized: Record<string, any> = {};
    const allowedKeys = new Set([
      'location', 'property_type', 'query', 'filters', 'title',
      'element_type', 'element_class', 'element_id', 'element_text',
      'x', 'y', 'scroll_top', 'scroll_height', 'client_height', 'scroll_percent',
      'search_session_id', 'timestamp'
    ]);

    for (const [key, value] of Object.entries(data)) {
      if (allowedKeys.has(key) && value !== null && value !== undefined) {
        if (typeof value === 'string') {
          sanitized[key] = value.slice(0, 500);
        } else if (typeof value === 'number') {
          sanitized[key] = value;
        } else if (typeof value === 'object') {
          sanitized[key] = value;
        }
        console.log(`✅ Kept key '${key}':`, value);
      } else {
        console.log(`❌ Filtered out key '${key}':`, value);
      }
    }

    console.log('🧼 Sanitized result:', sanitized);
    return sanitized;
  }

  private async processEventQueue(): Promise<void> {
    if (this.isProcessing || this.eventQueue.length === 0) return;
    
    this.isProcessing = true;
    
    try {
      const events = this.eventQueue.splice(0, 10); // Process up to 10 events at once
      
      for (const event of events) {
        await this.sendEvent(event);
        // Small delay to avoid overwhelming server
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    } finally {
      this.isProcessing = false;
      
      // Process remaining events if any
      if (this.eventQueue.length > 0) {
        setTimeout(() => this.processEventQueue(), 1000);
      }
    }
  }

  private async sendEvent(event: BehaviorEvent): Promise<void> {
    const token = localStorage.getItem('access_token');
    const endpoint = token ? '/behavior/track' : '/behavior/track-anonymous';

    console.log('📊 Tracking event:', event.event_type, event);

    const response = await fetch(`${this.apiBaseUrl}${endpoint}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` })
      },
      body: JSON.stringify(event)
    });

    if (response.ok) {
      console.log('✅ Event tracked successfully');
    } else {
      const errorText = await response.text();
      if (response.status === 429) {
        console.warn('⚠️ Rate limited, will retry later');
        // Re-add to queue for retry
        this.eventQueue.unshift(event);
      } else {
        console.warn('❌ Failed to track:', response.status, errorText);
      }
    }
  }

  public trackPropertyView(propertyId: number, propertyData?: Record<string, any>): void {
    const enhancedData = {
      ...propertyData,
      timestamp: Date.now(),
      session_duration: Date.now() - this.pageStartTime,
      scroll_depth: this.scrollDepth
    };
    this.trackEvent(EventType.PROPERTY_VIEW, enhancedData, propertyId);
  }

  public trackSearch(searchData: Record<string, any>): void {
    console.log('🔍 trackSearch called with:', searchData);
    const enhancedData = {
      ...searchData,
      search_session_id: crypto.randomUUID(),
      timestamp: Date.now()
    };
    console.log('🔍 Enhanced search data:', enhancedData);
    console.log('🔍 Calling trackEvent with EventType.SEARCH');
    this.trackEvent(EventType.SEARCH, enhancedData);
  }

  public trackSaveProperty(propertyId: number, propertyData?: Record<string, any>): void {
    const enhancedData = {
      ...propertyData,
      save_timestamp: Date.now(),
      time_to_save: Date.now() - this.pageStartTime,
      user_intent_score: this.calculateIntentScore()
    };
    this.trackEvent(EventType.SAVE_PROPERTY, enhancedData || {}, propertyId);
  }

  public trackWishlistAdd(propertyId: number, propertyData?: Record<string, any>): void {
    console.log('📊 trackWishlistAdd called with:', { propertyId, propertyData });
    const enhancedData = {
      ...propertyData,
      wishlist_timestamp: Date.now(),
      time_to_wishlist: Date.now() - this.pageStartTime,
      user_intent_score: this.calculateIntentScore()
    };
    console.log('📊 Calling trackEvent with WISHLIST_ADD');
    this.trackEvent(EventType.WISHLIST_ADD, enhancedData, propertyId);
  }

  public trackPropertyClick(propertyId: number, propertyData?: Record<string, any>): void {
    this.trackEvent(EventType.PROPERTY_CLICK, propertyData || {}, propertyId);
  }

  public trackFilterApply(filterData: Record<string, any>): void {
    this.trackEvent(EventType.FILTER_APPLY, filterData);
  }

  public trackSortChange(sortData: Record<string, any>): void {
    this.trackEvent(EventType.SORT_CHANGE, sortData);
  }

  public trackImageClick(propertyId: number, imageData?: Record<string, any>): void {
    this.trackEvent(EventType.IMAGE_CLICK, imageData || {}, propertyId);
  }

  public trackContactClick(propertyId: number, contactData?: Record<string, any>): void {
    this.trackEvent(EventType.CONTACT_CLICK, contactData || {}, propertyId);
  }

  public trackScheduleVisit(propertyId: number, visitData?: Record<string, any>): void {
    this.trackEvent(EventType.SCHEDULE_VISIT, visitData || {}, propertyId);
  }

  public trackDocumentClick(propertyId: number, documentData?: Record<string, any>): void {
    this.trackEvent(EventType.DOCUMENT_CLICK, documentData || {}, propertyId);
  }

  public trackBidPlace(propertyId: number, bidData?: Record<string, any>): void {
    this.trackEvent(EventType.BID_PLACE, bidData || {}, propertyId);
  }

  public trackAuctionRegister(propertyId: number, auctionData?: Record<string, any>): void {
    this.trackEvent(EventType.AUCTION_REGISTER, auctionData || {}, propertyId);
  }

  private calculateIntentScore(): number {
    // Simple intent scoring based on session behavior
    const sessionDuration = Date.now() - this.pageStartTime;
    const durationScore = Math.min(sessionDuration / 60000, 10); // Max 10 points for 1+ minutes
    const scrollScore = this.scrollDepth / 10; // Max 10 points for 100% scroll
    return Math.round(durationScore + scrollScore);
  }

  public disable(): void {
    this.isEnabled = false;
  }

  public enable(): void {
    this.isEnabled = true;
  }
}

export const behaviorTracker = new BehaviorTracker();

export const useBehaviorTracker = () => {
  return {
    trackPropertyView: behaviorTracker.trackPropertyView.bind(behaviorTracker),
    trackPropertyClick: behaviorTracker.trackPropertyClick.bind(behaviorTracker),
    trackSearch: behaviorTracker.trackSearch.bind(behaviorTracker),
    trackFilterApply: behaviorTracker.trackFilterApply.bind(behaviorTracker),
    trackSortChange: behaviorTracker.trackSortChange.bind(behaviorTracker),
    trackImageClick: behaviorTracker.trackImageClick.bind(behaviorTracker),
    trackContactClick: behaviorTracker.trackContactClick.bind(behaviorTracker),
    trackSaveProperty: behaviorTracker.trackSaveProperty.bind(behaviorTracker),
    trackWishlistAdd: behaviorTracker.trackWishlistAdd.bind(behaviorTracker),
    trackScheduleVisit: behaviorTracker.trackScheduleVisit.bind(behaviorTracker),
    trackDocumentClick: behaviorTracker.trackDocumentClick.bind(behaviorTracker),
    trackBidPlace: behaviorTracker.trackBidPlace.bind(behaviorTracker),
    trackAuctionRegister: behaviorTracker.trackAuctionRegister.bind(behaviorTracker),
    trackEvent: behaviorTracker.trackEvent.bind(behaviorTracker)
  };
};
