// Data Format Standardization

export const DATA_STANDARDS = {
  INTEREST_RATE: {
    PRECISION: 2, // 2 decimal places (e.g., 8.75%)
    MIN_RATE: 6.0,
    MAX_RATE: 15.0
  },
  PROCESSING_FEE: {
    TYPES: ['percentage', 'fixed'] as const,
    MAX_PERCENTAGE: 2.0, // 2% max
    MAX_FIXED: 50000 // ₹50,000 max
  },
  TENURE: {
    UNIT: 'years', // Always in years
    MIN_TENURE: 5,
    MAX_TENURE: 30,
    STEP: 5 // 5-year increments
  },
  LOAN_AMOUNT: {
    MIN_AMOUNT: 100000, // ₹1 Lakh
    MAX_AMOUNT: 100000000, // ₹10 Crores
    CURRENCY: 'INR'
  }
};

// Rate formatting utilities
export const formatInterestRate = (rate: number): number => {
  return Math.round(rate * 100) / 100; // 2 decimal precision
};

export const validateInterestRate = (rate: number): boolean => {
  return rate >= DATA_STANDARDS.INTEREST_RATE.MIN_RATE && 
         rate <= DATA_STANDARDS.INTEREST_RATE.MAX_RATE;
};

// Processing fee utilities
export const formatProcessingFee = (fee: { type: 'percentage' | 'fixed'; value: number; gst: boolean }) => {
  if (fee.type === 'percentage') {
    return `${fee.value}%${fee.gst ? ' + GST' : ''}`;
  }
  return `₹${fee.value.toLocaleString('en-IN')}${fee.gst ? ' + GST' : ''}`;
};

export const calculateProcessingFee = (
  loanAmount: number, 
  fee: { type: 'percentage' | 'fixed'; value: number; gst: boolean }
): number => {
  let feeAmount = 0;
  
  if (fee.type === 'percentage') {
    feeAmount = (loanAmount * fee.value) / 100;
  } else {
    feeAmount = fee.value;
  }
  
  if (fee.gst) {
    feeAmount = feeAmount * 1.18; // 18% GST
  }
  
  return Math.round(feeAmount);
};

// Tenure utilities
export const convertMonthsToYears = (months: number): number => {
  return Math.round(months / 12);
};

export const convertYearsToMonths = (years: number): number => {
  return years * 12;
};

export const validateTenure = (years: number): boolean => {
  return years >= DATA_STANDARDS.TENURE.MIN_TENURE && 
         years <= DATA_STANDARDS.TENURE.MAX_TENURE;
};

// Amount formatting
export const formatCurrency = (amount: number): string => {
  if (amount >= 10000000) { // 1 Crore+
    return `₹${(amount / 10000000).toFixed(2)} Cr`;
  } else if (amount >= 100000) { // 1 Lakh+
    return `₹${(amount / 100000).toFixed(2)} L`;
  }
  return `₹${amount.toLocaleString('en-IN')}`;
};

// Data transformation utilities
export const transformBankDataFromAPI = (apiData: any) => {
  return {
    ...apiData,
    rates: {
      ...apiData.rates,
      minRate: formatInterestRate(apiData.rates.minRate),
      maxRate: formatInterestRate(apiData.rates.maxRate),
      rateByTenure: Object.fromEntries(
        Object.entries(apiData.rates.rateByTenure).map(([tenure, rate]) => [
          tenure,
          formatInterestRate(rate as number)
        ])
      )
    }
  };
};