// Documents API integration
export interface PropertyDocument {
  id: number;
  property_id: number;
  doc_type: string;
  file_url: string;
  user_id: string;
  created_at: string;
  created_by?: string;
  updated_by?: string;
  updated_time?: string;
}

export interface DocumentsResponse {
  documents: PropertyDocument[];
  total_count: number;
}

const getAuthHeaders = (): Record<string, string> => {
  const token = localStorage.getItem('access_token') || localStorage.getItem('token');
  return token ? { 'Authorization': `Bearer ${token}` } : {};
};

export const documentsApi = {
  // Upload document
  uploadDocument: async (
    file: File,
    propertyId: string | number,
    userId: string,
    documentType: string
  ): Promise<PropertyDocument> => {
    console.log('📄 documentsApi.uploadDocument called with:', {
      fileName: file.name,
      propertyId,
      userId,
      documentType
    });
    
    const formData = new FormData();
    formData.append('file', file);
    formData.append('property_id', propertyId.toString());
    formData.append('user_id', userId);
    formData.append('doc_type', documentType);

    console.log('📄 Sending to: http://localhost:8090/api/documents/upload');
    
    const response = await fetch('http://localhost:8090/api/documents/upload', {
      method: 'POST',
      headers: getAuthHeaders(),
      body: formData
    });

    console.log('📄 Upload response:', response.status, response.statusText);
    
    if (!response.ok) {
      const error = await response.text();
      console.error('❌ Upload failed:', error);
      throw new Error(`Upload failed: ${response.status} ${error}`);
    }

    const result = await response.json();
    console.log('✅ Upload successful:', result);
    return result;
  },

  // Get documents for property
  getPropertyDocuments: async (propertyId: number): Promise<DocumentsResponse> => {
    console.log('🔍 Fetching documents for property ID:', propertyId);
    
    // Try multiple endpoints since documents exist in DB
    const endpoints = [
      `http://localhost:8090/api/documents/property/${propertyId}`,
      `http://localhost:8090/api/property-documents/${propertyId}`,
      `http://localhost:8090/api/documents?property_id=${propertyId}`
    ];
    
    for (const endpoint of endpoints) {
      try {
        console.log(`📡 Trying: ${endpoint}`);
        const response = await fetch(endpoint, {
          headers: getAuthHeaders()
        });

        console.log(`📥 Response:`, response.status, response.statusText);
        
        if (response.ok) {
          const data = await response.json();
          console.log(`✅ Documents found:`, data);
          
          // Handle different response formats
          if (data.documents) return data;
          if (Array.isArray(data)) return { documents: data, total_count: data.length };
          if (data.results) return { documents: data.results, total_count: data.results.length };
        }
      } catch (error) {
        console.log(`❌ Error with ${endpoint}:`, error);
      }
    }
    
    console.log(`❌ No documents found for property ${propertyId}`);
    return { documents: [], total_count: 0 };
  },

  // Generate download URL
  getDownloadUrl: async (documentId: number): Promise<{ download_url: string; expires_in_hours: number }> => {
    try {
      const response = await fetch(`/api/documents/${documentId}/download`, {
        headers: getAuthHeaders()
      });

      if (response.ok) {
        return response.json();
      }
    } catch (error) {
      console.log('Download API not available, using direct URL');
    }

    return {
      download_url: `https://nalpropertysa.blob.core.windows.net/documents/mock/document_${documentId}.pdf`,
      expires_in_hours: 24
    };
  },

  // Delete document
  deleteDocument: async (documentId: number): Promise<void> => {
    const response = await fetch(`/api/documents/${documentId}`, {
      method: 'DELETE',
      headers: getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Delete failed');
    }
  }
};