interface ErrorDetails {
  message: string;
  code?: string;
  context?: string;
}

class ErrorHandler {
  private static instance: ErrorHandler;
  
  static getInstance(): ErrorHandler {
    if (!ErrorHandler.instance) {
      ErrorHandler.instance = new ErrorHandler();
    }
    return ErrorHandler.instance;
  }

  handleError(error: unknown, context?: string): ErrorDetails {
    let message = 'An unexpected error occurred';
    let code = 'UNKNOWN_ERROR';

    if (error instanceof Error) {
      message = error.message;
      
      // Network errors
      if (error.message.includes('fetch')) {
        message = 'Network connection failed. Please check your internet connection.';
        code = 'NETWORK_ERROR';
      }
      
      // API errors
      if (error.message.includes('HTTP error')) {
        message = 'Service temporarily unavailable. Please try again later.';
        code = 'API_ERROR';
      }
      
      // Validation errors
      if (error.message.includes('validation') || error.message.includes('required')) {
        message = 'Please check your input and try again.';
        code = 'VALIDATION_ERROR';
      }
    }

    // Log error for debugging (in production, send to monitoring service)
    console.error(`[${code}] ${context || 'Unknown context'}:`, error);

    return { message, code, context };
  }

  async withErrorHandling<T>(
    operation: () => Promise<T>,
    context: string,
    fallback?: T
  ): Promise<T> {
    try {
      return await operation();
    } catch (error) {
      const errorDetails = this.handleError(error, context);
      
      if (fallback !== undefined) {
        console.warn(`Using fallback for ${context}:`, errorDetails.message);
        return fallback;
      }
      
      throw new Error(errorDetails.message);
    }
  }

  isRetryableError(error: unknown): boolean {
    if (error instanceof Error) {
      return error.message.includes('fetch') || 
             error.message.includes('timeout') ||
             error.message.includes('HTTP error! status: 5');
    }
    return false;
  }

  async withRetry<T>(
    operation: () => Promise<T>,
    maxRetries: number = 3,
    delay: number = 1000
  ): Promise<T> {
    let lastError: unknown;
    
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
      try {
        return await operation();
      } catch (error) {
        lastError = error;
        
        if (attempt === maxRetries || !this.isRetryableError(error)) {
          throw error;
        }
        
        await new Promise(resolve => setTimeout(resolve, delay * attempt));
      }
    }
    
    throw lastError;
  }
}

export const errorHandler = ErrorHandler.getInstance();

// Utility functions for common error scenarios
export const safeAsync = async <T>(
  operation: () => Promise<T>,
  fallback: T,
  context?: string
): Promise<T> => {
  return errorHandler.withErrorHandling(operation, context || 'async operation', fallback);
};

export const withRetry = <T>(
  operation: () => Promise<T>,
  maxRetries?: number,
  delay?: number
): Promise<T> => {
  return errorHandler.withRetry(operation, maxRetries, delay);
};