// Flood Risk Calculation Utilities

import { FloodRiskData, FloodRiskCalculation } from '../types/gee';

/**
 * Calculate flood risk score from raw GEE data
 * Formula: Weighted average of elevation, proximity, history, and occurrence
 */
export function calculateFloodScore(
  elevation: number,
  waterDistance: number,
  historicalFloods: number,
  waterOccurrence: number
): FloodRiskCalculation {
  // Normalize elevation (0-100m range, lower = higher risk)
  const elevationScore = Math.max(0, Math.min(10, (100 - elevation) / 10));
  
  // Normalize water proximity (0-5000m range, closer = higher risk)
  const proximityScore = Math.max(0, Math.min(10, (5000 - waterDistance) / 500));
  
  // Historical floods (0-5 events, more = higher risk)
  const historyScore = Math.min(10, historicalFloods * 2);
  
  // Water occurrence frequency (0-100%, higher = higher risk)
  const occurrenceScore = waterOccurrence / 10;
  
  // Weighted final score
  const finalScore = (
    elevationScore * 0.4 +
    proximityScore * 0.3 +
    historyScore * 0.2 +
    occurrenceScore * 0.1
  );
  
  return {
    elevationScore,
    proximityScore,
    historyScore,
    occurrenceScore,
    finalScore: Math.round(finalScore * 10) / 10
  };
}

/**
 * Classify risk level based on score
 */
export function classifyRiskLevel(score: number): 'low' | 'medium' | 'high' {
  if (score <= 3) return 'low';
  if (score <= 6) return 'medium';
  return 'high';
}

/**
 * Get color for risk level
 */
export function getFloodColor(score: number): string {
  if (score <= 3) return '#10b981'; // green
  if (score <= 6) return '#f59e0b'; // yellow
  return '#ef4444'; // red
}

/**
 * Get background color for risk badge
 */
export function getFloodBgColor(score: number): string {
  if (score <= 3) return 'bg-green-100 text-green-800 border-green-300';
  if (score <= 6) return 'bg-yellow-100 text-yellow-800 border-yellow-300';
  return 'bg-red-100 text-red-800 border-red-300';
}

/**
 * Format flood risk data for display
 */
export function formatFloodData(
  elevation: number,
  waterDistance: number,
  historicalFloods: number,
  waterOccurrence: number
): FloodRiskData {
  const calculation = calculateFloodScore(elevation, waterDistance, historicalFloods, waterOccurrence);
  
  return {
    score: calculation.finalScore,
    level: classifyRiskLevel(calculation.finalScore),
    elevation: Math.round(elevation),
    waterDistance: Math.round(waterDistance),
    historicalFloods,
    waterOccurrence: Math.round(waterOccurrence),
    lastUpdated: new Date().toISOString()
  };
}

/**
 * Generate cache key for location
 */
export function getFloodCacheKey(lat: number, lng: number): string {
  return `flood_${lat.toFixed(4)}_${lng.toFixed(4)}`;
}

/**
 * Check if cached data is still valid (1 hour)
 */
export function isCacheValid(timestamp: number): boolean {
  const ONE_HOUR = 60 * 60 * 1000;
  return Date.now() - timestamp < ONE_HOUR;
}
