import { ListingFormData } from '../types/listing';

export interface ValidationResult {
  isValid: boolean;
  errors: string[];
  missingFields: string[];
}

export const validateListingForm = (formData: Partial<ListingFormData>): ValidationResult => {
  const errors: string[] = [];
  const missingFields: string[] = [];

  // Basic required fields
  const basicFields = {
    ownerName: 'Owner Name',
    mobile: 'Mobile Number',
    email: 'Email Address',
    title: 'Property Title',
    description: 'Property Description',
    listingType: 'Listing Type',
    propertyType: 'Property Type'
  };

  // Property details required fields
  const propertyFields = {
    builtUpArea: 'Built-up Area',
    bedrooms: 'Bedrooms',
    bathrooms: 'Bathrooms',
    facing: 'Property Facing',
    furnishing: 'Furnishing Status',
    propertyAge: 'Property Age',
    gatedCommunity: 'Gated Community'
  };

  // Location required fields
  const locationFields = {
    address: 'Property Address',
    city: 'City',
    locality: 'Locality',
    state: 'State',
    pincode: 'Pincode'
  };

  // Check basic fields
  Object.entries(basicFields).forEach(([key, label]) => {
    const value = formData[key as keyof ListingFormData];
    if (!value || (typeof value === 'string' && value.trim() === '')) {
      missingFields.push(key);
      errors.push(`${label} is required`);
    }
  });

  // Check property fields
  Object.entries(propertyFields).forEach(([key, label]) => {
    const value = formData[key as keyof ListingFormData];
    if (!value || (typeof value === 'string' && value.trim() === '')) {
      missingFields.push(key);
      errors.push(`${label} is required`);
    }
  });

  // Check location fields
  Object.entries(locationFields).forEach(([key, label]) => {
    const value = formData[key as keyof ListingFormData];
    if (!value || (typeof value === 'string' && value.trim() === '')) {
      missingFields.push(key);
      errors.push(`${label} is required`);
    }
  });

  // Validate pricing based on listing type
  if (formData.listingType === 'sell' && !formData.salePrice) {
    missingFields.push('salePrice');
    errors.push('Sale Price is required for sell listings');
  }

  if (formData.listingType === 'rent' && !formData.monthlyRent) {
    missingFields.push('monthlyRent');
    errors.push('Monthly Rent is required for rent listings');
  }

  if (formData.listingType === 'bidding' && !formData.startingBidPrice) {
    missingFields.push('startingBidPrice');
    errors.push('Starting Bid Price is required for bidding listings');
  }

  if (formData.listingType === 'urgent-sale' && !formData.actualPrice) {
    missingFields.push('actualPrice');
    errors.push('Actual Price is required for urgent sale listings');
  }

  if (formData.listingType === 'lease') {
    const leaseFields = ['leaseType', 'leaseDuration', 'monthlyLeaseAmount', 'leaseStartDate', 'possessionAvailability'];
    leaseFields.forEach(field => {
      const value = formData[field as keyof ListingFormData];
      if (!value || (typeof value === 'string' && value.trim() === '')) {
        missingFields.push(field);
        errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required for lease listings`);
      }
    });
  }

  // Validate PG-specific fields
  if (formData.propertyType === 'pg-coliving') {
    const pgFields = ['accommodationType', 'totalBedsAvailable', 'perBedRent', 'genderPreference', 'availabilityDate'];
    pgFields.forEach(field => {
      const value = formData[field as keyof ListingFormData];
      if (!value || (typeof value === 'string' && value.trim() === '')) {
        missingFields.push(field);
        errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required for PG/Co-living properties`);
      }
    });
  }

  // Validate mobile number format
  if (formData.mobile && !/^[6-9]\d{9}$/.test(formData.mobile)) {
    errors.push('Please enter a valid 10-digit Indian mobile number starting with 6-9');
  }

  // Validate email format
  if (formData.email && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
    errors.push('Please enter a valid email address');
  }

  // Validate pincode format
  if (formData.pincode && !/^[1-9][0-9]{5}$/.test(formData.pincode)) {
    errors.push('Please enter a valid 6-digit Indian pincode');
  }

  // Validate numeric fields
  if (formData.builtUpArea && formData.builtUpArea <= 0) {
    errors.push('Built-up area must be greater than 0');
  }

  if (formData.salePrice && formData.salePrice <= 0) {
    errors.push('Sale price must be greater than 0');
  }

  if (formData.monthlyRent && formData.monthlyRent <= 0) {
    errors.push('Monthly rent must be greater than 0');
  }

  return {
    isValid: errors.length === 0,
    errors,
    missingFields
  };
};

export const validateStep = (step: number, formData: Partial<ListingFormData>): ValidationResult => {
  const errors: string[] = [];
  const missingFields: string[] = [];

  switch (step) {
    case 0: // Basic Info
      const basicFields = ['ownerName', 'mobile', 'email', 'title', 'description', 'listingType', 'propertyType'];
      basicFields.forEach(field => {
        const value = formData[field as keyof ListingFormData];
        if (!value || (typeof value === 'string' && value.trim() === '')) {
          missingFields.push(field);
          errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required`);
        }
      });
      break;

    case 1: // Property Details & Pricing
      const propertyFields = ['builtUpArea', 'bedrooms', 'bathrooms', 'facing', 'furnishing', 'propertyAge', 'gatedCommunity'];
      propertyFields.forEach(field => {
        const value = formData[field as keyof ListingFormData];
        if (!value || (typeof value === 'string' && value.trim() === '')) {
          missingFields.push(field);
          errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required`);
        }
      });

      // Validate pricing based on listing type
      if (formData.listingType === 'sell' && !formData.salePrice) {
        missingFields.push('salePrice');
        errors.push('Sale Price is required');
      }
      if (formData.listingType === 'rent' && !formData.monthlyRent) {
        missingFields.push('monthlyRent');
        errors.push('Monthly Rent is required');
      }
      if (formData.listingType === 'bidding' && !formData.startingBidPrice) {
        missingFields.push('startingBidPrice');
        errors.push('Starting Bid Price is required');
      }
      if (formData.listingType === 'urgent-sale' && !formData.actualPrice) {
        missingFields.push('actualPrice');
        errors.push('Actual Price is required');
      }
      if (formData.listingType === 'lease') {
        const leaseFields = ['leaseType', 'leaseDuration', 'monthlyLeaseAmount', 'leaseStartDate', 'possessionAvailability'];
        leaseFields.forEach(field => {
          const value = formData[field as keyof ListingFormData];
          if (!value || (typeof value === 'string' && value.trim() === '')) {
            missingFields.push(field);
            errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required`);
          }
        });
      }

      if (formData.propertyType === 'pg-coliving') {
        const pgFields = ['accommodationType', 'totalBedsAvailable', 'perBedRent', 'genderPreference', 'availabilityDate'];
        pgFields.forEach(field => {
          const value = formData[field as keyof ListingFormData];
          if (!value || (typeof value === 'string' && value.trim() === '')) {
            missingFields.push(field);
            errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required`);
          }
        });
      }
      break;

    case 2: // Location & Media
      const locationFields = ['address', 'city', 'locality', 'state', 'pincode'];
      locationFields.forEach(field => {
        const value = formData[field as keyof ListingFormData];
        if (!value || (typeof value === 'string' && value.trim() === '')) {
          missingFields.push(field);
          errors.push(`${field.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())} is required`);
        }
      });
      break;

    case 3: // Documents (optional)
      // Documents are optional, no validation needed
      break;

    case 4: // Review & Submit
      return validateListingForm(formData);
  }

  return {
    isValid: errors.length === 0,
    errors,
    missingFields
  };
};
