import jsPDF from 'jspdf';

interface ValuationData {
  formData: {
    location: string;
    property_type: string;
    area: string;
    bedrooms: number;
    bathrooms: number;
    property_age: number;
    floor_number: number;
    total_floors: number;
    furnishing: string;
    facing: string;
  };
  result: {
    estimated_value: number;
    price_per_sqft: number;
    valuation_range_min: number;
    valuation_range_max: number;
    confidence_score: number;
    confidence_level: string;
    comparables?: Array<{
      address: string;
      price: number;
      area: number;
      price_per_sqft: number;
      distance_km: number;
    }>;
    location_city?: string;
    location_locality?: string;
  };
}

export const generateValuationPDF = async (data: ValuationData) => {
  const doc = new jsPDF();
  const pageWidth = doc.internal.pageSize.getWidth();
  const pageHeight = doc.internal.pageSize.getHeight();
  
  // Colors
  const primaryColor = '#4f46e5';
  const secondaryColor = '#8b5cf6';
  const textColor = '#1f2937';
  const lightGray = '#f3f4f6';
  
  // Header with gradient effect
  doc.setFillColor(79, 70, 229);
  doc.rect(0, 0, pageWidth, 40, 'F');
  
  // Add NAL Logo
  try {
    const logoImg = new Image();
    logoImg.src = '/favicon.png';
    await new Promise((resolve, reject) => {
      logoImg.onload = resolve;
      logoImg.onerror = reject;
    });
    doc.addImage(logoImg, 'PNG', 15, 8, 25, 25);
  } catch (error) {
    console.error('Failed to load logo:', error);
  }
  
  // NAL text next to logo
  doc.setTextColor(255, 255, 255);
  doc.setFontSize(20);
  doc.setFont('helvetica', 'bold');
  doc.text('NAL India', 45, 20);
  doc.setFontSize(9);
  doc.setFont('helvetica', 'normal');
  doc.text('New Age Land', 45, 27);
  
  // Report title
  doc.setFontSize(16);
  doc.setFont('helvetica', 'bold');
  doc.text('Property Valuation Report', pageWidth - 15, 25, { align: 'right' });
  
  // Date
  doc.setFontSize(9);
  doc.setFont('helvetica', 'normal');
  doc.text(`Generated: ${new Date().toLocaleDateString('en-IN')}`, pageWidth - 15, 32, { align: 'right' });
  
  let yPos = 50;
  
  // Property Details Section
  doc.setTextColor(textColor);
  doc.setFontSize(14);
  doc.setFont('helvetica', 'bold');
  doc.text('Property Details', 15, yPos);
  yPos += 10;
  
  // Property details box
  doc.setFillColor(243, 244, 246);
  doc.roundedRect(15, yPos, pageWidth - 30, 50, 3, 3, 'F');
  
  doc.setFontSize(10);
  doc.setFont('helvetica', 'normal');
  yPos += 8;
  
  const details = [
    ['Location:', data.formData.location],
    ['Property Type:', data.formData.property_type.charAt(0).toUpperCase() + data.formData.property_type.slice(1)],
    ['Area:', `${data.formData.area} sq ft`],
    ['Configuration:', `${data.formData.bedrooms} BHK, ${data.formData.bathrooms} Bathrooms`],
    ['Age:', `${data.formData.property_age} years`],
    ['Floor:', `${data.formData.floor_number} of ${data.formData.total_floors}`],
    ['Furnishing:', data.formData.furnishing.replace('_', ' ').charAt(0).toUpperCase() + data.formData.furnishing.slice(1).replace('_', ' ')],
    ['Facing:', data.formData.facing.charAt(0).toUpperCase() + data.formData.facing.slice(1)]
  ];
  
  details.forEach(([label, value]) => {
    doc.setFont('helvetica', 'bold');
    doc.text(label, 20, yPos);
    doc.setFont('helvetica', 'normal');
    doc.text(value, 70, yPos);
    yPos += 6;
  });
  
  yPos += 10;
  
  // Valuation Results Section
  doc.setFontSize(14);
  doc.setFont('helvetica', 'bold');
  doc.text('Valuation Results', 15, yPos);
  yPos += 10;
  
  // Main valuation box
  doc.setFillColor(79, 70, 229);
  doc.roundedRect(15, yPos, pageWidth - 30, 35, 3, 3, 'F');
  
  doc.setTextColor(255, 255, 255);
  doc.setFontSize(12);
  doc.text('Estimated Property Value', 20, yPos + 10);
  doc.setFontSize(20);
  doc.setFont('helvetica', 'bold');
  doc.text(`Rs ${(data.result.estimated_value / 100000).toFixed(2)} Lakhs`, 20, yPos + 22);
  doc.setFontSize(10);
  doc.setFont('helvetica', 'normal');
  doc.text(`Rs ${data.result.price_per_sqft.toFixed(0)} per sq ft`, 20, yPos + 30);
  
  yPos += 45;
  
  // Value range and confidence
  doc.setTextColor(textColor);
  doc.setFillColor(243, 244, 246);
  doc.roundedRect(15, yPos, (pageWidth - 35) / 2, 25, 3, 3, 'F');
  doc.roundedRect(15 + (pageWidth - 35) / 2 + 5, yPos, (pageWidth - 35) / 2, 25, 3, 3, 'F');
  
  doc.setFontSize(10);
  doc.setFont('helvetica', 'bold');
  doc.text('Value Range', 20, yPos + 8);
  doc.setFont('helvetica', 'normal');
  doc.setFontSize(9);
  doc.text(`Rs ${(data.result.valuation_range_min / 100000).toFixed(2)}L - Rs ${(data.result.valuation_range_max / 100000).toFixed(2)}L`, 20, yPos + 18);
  
  doc.setFont('helvetica', 'bold');
  doc.setFontSize(10);
  doc.text('Confidence Score', 20 + (pageWidth - 35) / 2 + 5, yPos + 8);
  doc.setFont('helvetica', 'normal');
  doc.setFontSize(9);
  doc.text(`${data.result.confidence_score}% (${data.result.confidence_level})`, 20 + (pageWidth - 35) / 2 + 5, yPos + 18);
  
  yPos += 35;
  
  // Comparable Properties
  if (data.result.comparables && data.result.comparables.length > 0) {
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('Comparable Properties', 15, yPos);
    yPos += 8;
    
    data.result.comparables.forEach((comp, idx) => {
      if (yPos > pageHeight - 40) {
        doc.addPage();
        yPos = 20;
      }
      
      doc.setFillColor(243, 244, 246);
      doc.roundedRect(15, yPos, pageWidth - 30, 20, 3, 3, 'F');
      
      doc.setFontSize(10);
      doc.setFont('helvetica', 'bold');
      doc.text(`${idx + 1}. ${comp.address}`, 20, yPos + 7);
      doc.setFont('helvetica', 'normal');
      doc.setFontSize(9);
      doc.text(`${comp.area} sq ft | ${comp.distance_km} km away`, 20, yPos + 13);
      doc.setFont('helvetica', 'bold');
      doc.text(`Rs ${(comp.price / 100000).toFixed(2)}L`, pageWidth - 20, yPos + 10, { align: 'right' });
      doc.setFont('helvetica', 'normal');
      doc.text(`Rs ${comp.price_per_sqft}/sq ft`, pageWidth - 20, yPos + 16, { align: 'right' });
      
      yPos += 25;
    });
  }
  
  // Footer
  doc.setFontSize(8);
  doc.setTextColor(128, 128, 128);
  doc.text('This is a computer-generated report. For certified valuation, please contact NAL India.', pageWidth / 2, pageHeight - 15, { align: 'center' });
  doc.text('NAL India | New Age Land | www.nalindia.com | support@nalindia.com', pageWidth / 2, pageHeight - 10, { align: 'center' });
  
  // Save PDF
  const fileName = `NAL_Valuation_Report_${data.formData.location.replace(/[^a-z0-9]/gi, '_')}_${new Date().getTime()}.pdf`;
  doc.save(fileName);
};
