/**
 * Secure Google Maps API integration with error handling
 */

interface GoogleMapsConfig {
  apiKey: string;
  libraries: string[];
  region?: string;
  language?: string;
}

class GoogleMapsService {
  private static instance: GoogleMapsService;
  private isLoaded = false;
  private loadPromise: Promise<void> | null = null;
  private config: GoogleMapsConfig;

  private constructor() {
    const apiKey = import.meta.env.VITE_GOOGLE_MAPS_API_KEY || '';

    this.config = {
      apiKey,
      libraries: ['places', 'geometry'],
      region: 'IN',
      language: 'en'
    };
  }

  static getInstance(): GoogleMapsService {
    if (!GoogleMapsService.instance) {
      GoogleMapsService.instance = new GoogleMapsService();
    }
    return GoogleMapsService.instance;
  }

  async loadGoogleMaps(): Promise<void> {
    if (this.isLoaded) return;
    if (this.loadPromise) return this.loadPromise;

    this.loadPromise = new Promise((resolve, reject) => {
      // Check if already loaded
      if (window.google?.maps) {
        this.isLoaded = true;
        resolve();
        return;
      }

      // Skip loading if no API key
      if (!this.config.apiKey) {
        console.warn('Google Maps API key not configured. Maps features will be disabled.');
        resolve();
        return;
      }

      const script = document.createElement('script');
      script.async = true;
      script.defer = true;
      
      const params = new URLSearchParams({
        key: this.config.apiKey,
        libraries: this.config.libraries.join(','),
        region: this.config.region || 'IN',
        language: this.config.language || 'en'
      });

      script.src = `https://maps.googleapis.com/maps/api/js?${params.toString()}`;
      
      script.onload = () => {
        this.isLoaded = true;
        resolve();
      };

      script.onerror = (error) => {
        console.error('Failed to load Google Maps API:', error);
        reject(new Error('Google Maps API failed to load'));
      };

      document.head.appendChild(script);
    });

    return this.loadPromise;
  }

  isGoogleMapsLoaded(): boolean {
    return this.isLoaded && !!window.google?.maps;
  }

  getApiKey(): string {
    return this.config.apiKey;
  }
}

export const googleMapsService = GoogleMapsService.getInstance();
export default googleMapsService;