/**
 * Professional heatmap algorithms - NO AMATEUR HOUR BULLSHIT
 */

import { Property, HeatmapPoint } from '../components/heatmap/types';

interface GridConfig {
  size: number;
  radius: number;
  minPoints: number;
}

export class HeatmapAlgorithms {
  /**
   * Get proper grid size based on zoom level - NOT FIXED 1KM GARBAGE
   */
  static getGridConfig(zoomLevel: number): GridConfig {
    if (zoomLevel >= 15) {
      return { size: 0.001, radius: 0.002, minPoints: 2 }; // ~100m grid for street level
    }
    if (zoomLevel >= 12) {
      return { size: 0.005, radius: 0.01, minPoints: 3 }; // ~500m grid for neighborhood
    }
    if (zoomLevel >= 10) {
      return { size: 0.01, radius: 0.02, minPoints: 5 }; // ~1km grid for city
    }
    return { size: 0.02, radius: 0.04, minPoints: 8 }; // ~2km grid for region
  }

  /**
   * Statistical density calculation with Gaussian kernel - NOT SIMPLE COUNTING
   */
  static calculateDensityHeatmap(properties: Property[], zoomLevel: number = 12): HeatmapPoint[] {
    if (properties.length === 0) return [];

    // For small datasets, show individual properties
    if (properties.length <= 5) {
      return properties.map(property => ({
        lat: property.latitude,
        lng: property.longitude,
        intensity: 0.7,
        propertyCount: 1
      }));
    }

    const config = this.getGridConfig(zoomLevel);
    const densityMap = new Map<string, { 
      lat: number; 
      lng: number; 
      count: number; 
      weightedSum: number;
      properties: Property[];
    }>();

    // Build density grid with Gaussian weighting
    properties.forEach(property => {
      const gridLat = Math.round(property.latitude / config.size) * config.size;
      const gridLng = Math.round(property.longitude / config.size) * config.size;
      
      // Apply Gaussian influence to nearby grid cells
      for (let latOffset = -1; latOffset <= 1; latOffset++) {
        for (let lngOffset = -1; lngOffset <= 1; lngOffset++) {
          const targetLat = gridLat + (latOffset * config.size);
          const targetLng = gridLng + (lngOffset * config.size);
          const key = `${targetLat.toFixed(6)},${targetLng.toFixed(6)}`;
          
          const distance = this.haversineDistance(
            property.latitude, property.longitude,
            targetLat, targetLng
          );
          
          // Gaussian weight based on distance
          const weight = Math.exp(-(distance * distance) / (2 * config.radius * config.radius));
          
          if (weight > 0.1) { // Only significant influences
            const existing = densityMap.get(key) || { 
              lat: targetLat, 
              lng: targetLng, 
              count: 0, 
              weightedSum: 0,
              properties: []
            };
            
            existing.count += weight;
            existing.weightedSum += weight;
            if (latOffset === 0 && lngOffset === 0) {
              existing.properties.push(property);
            }
            
            densityMap.set(key, existing);
          }
        }
      }
    });

    // Filter and normalize - reduce minimum points for small datasets
    const minPoints = Math.max(1, Math.min(config.minPoints, properties.length / 3));
    const validCells = Array.from(densityMap.values())
      .filter(cell => cell.count >= minPoints);

    if (validCells.length === 0) {
      // Fallback: show individual properties
      return properties.map(property => ({
        lat: property.latitude,
        lng: property.longitude,
        intensity: 0.6,
        propertyCount: 1
      }));
    }

    // Simple normalization for small datasets
    const maxCount = Math.max(...validCells.map(cell => cell.count));

    return validCells.map(cell => {
      const intensity = Math.max(0.3, Math.min(1, cell.count / maxCount));

      return {
        lat: cell.lat,
        lng: cell.lng,
        intensity,
        propertyCount: cell.properties.length
      };
    });
  }

  /**
   * Dynamic price analysis with percentile-based ranges - NOT FIXED 5CR CAP
   */
  static calculatePriceHeatmap(properties: Property[]): HeatmapPoint[] {
    if (properties.length === 0) return [];

    // Get actual prices from properties
    const prices = properties
      .map(p => p.sale_price || (p.monthly_rent * 12) || 0)
      .filter(price => price > 0)
      .sort((a, b) => a - b);

    if (prices.length === 0) return [];

    // Calculate dynamic percentiles
    const p25 = this.percentile(prices, 25);
    const p50 = this.percentile(prices, 50);
    const p75 = this.percentile(prices, 75);
    const p90 = this.percentile(prices, 90);

    return properties
      .filter(p => (p.sale_price || p.monthly_rent) > 0)
      .map(property => {
        const price = property.sale_price || (property.monthly_rent * 12);
        
        let intensity: number;
        if (price <= p25) intensity = 0.2;      // Low
        else if (price <= p50) intensity = 0.4; // Below median
        else if (price <= p75) intensity = 0.6; // Above median
        else if (price <= p90) intensity = 0.8; // High
        else intensity = 1.0;                   // Premium

        return {
          lat: property.latitude,
          lng: property.longitude,
          intensity,
          price
        };
      });
  }

  /**
   * Price per sqft normalization for fair comparison
   */
  static calculatePricePerSqftHeatmap(properties: Property[]): HeatmapPoint[] {
    const validProperties = properties.filter(p => 
      (p.sale_price || p.monthly_rent) > 0 && 
      (p.built_up_area || p.carpet_area || p.super_area) > 0
    );

    if (validProperties.length === 0) return [];

    const pricePerSqft = validProperties.map(p => {
      const price = p.sale_price || (p.monthly_rent * 12);
      const area = p.built_up_area || p.carpet_area || p.super_area || 1;
      return price / area;
    }).sort((a, b) => a - b);

    const p25 = this.percentile(pricePerSqft, 25);
    const p75 = this.percentile(pricePerSqft, 75);
    const p95 = this.percentile(pricePerSqft, 95);

    return validProperties.map(property => {
      const price = property.sale_price || (property.monthly_rent * 12);
      const area = property.built_up_area || property.carpet_area || property.super_area || 1;
      const psf = price / area;

      let intensity: number;
      if (psf <= p25) intensity = 0.25;
      else if (psf <= p75) intensity = 0.6;
      else if (psf <= p95) intensity = 0.85;
      else intensity = 1.0;

      return {
        lat: property.latitude,
        lng: property.longitude,
        intensity,
        price: psf,
        propertyCount: 1
      };
    });
  }

  /**
   * Haversine distance calculation
   */
  private static haversineDistance(lat1: number, lng1: number, lat2: number, lng2: number): number {
    const R = 6371; // Earth's radius in km
    const dLat = this.toRadians(lat2 - lat1);
    const dLng = this.toRadians(lng2 - lng1);
    
    const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
              Math.cos(this.toRadians(lat1)) * Math.cos(this.toRadians(lat2)) *
              Math.sin(dLng / 2) * Math.sin(dLng / 2);
    
    return 2 * R * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  }

  /**
   * Calculate percentile
   */
  private static percentile(sortedArray: number[], percentile: number): number {
    const index = (percentile / 100) * (sortedArray.length - 1);
    const lower = Math.floor(index);
    const upper = Math.ceil(index);
    
    if (lower === upper) return sortedArray[lower];
    
    const weight = index - lower;
    return sortedArray[lower] * (1 - weight) + sortedArray[upper] * weight;
  }

  private static toRadians(degrees: number): number {
    return degrees * (Math.PI / 180);
  }
}

export default HeatmapAlgorithms;