// Infrastructure & Accessibility Calculator

import { InfrastructureData } from '../types/gee';

// Calculate accessibility score based on distances
export const calculateAccessibilityScore = (
  highwayDist: number,
  metroDist: number | null,
  airportDist: number,
  itParkDist: number | null,
  hospitalDist: number,
  schoolDist: number
): number => {
  // Weighted scoring (0-10 scale)
  const highwayScore = Math.max(0, 10 - (highwayDist / 2)); // 2km = 0 points
  const metroScore = metroDist ? Math.max(0, 10 - (metroDist / 1)) : 5; // 1km = 0 points, null = 5
  const airportScore = Math.max(0, 10 - (airportDist / 5)); // 5km = 0 points
  const itParkScore = itParkDist ? Math.max(0, 10 - (itParkDist / 3)) : 5; // 3km = 0 points
  const hospitalScore = Math.max(0, 10 - (hospitalDist / 2)); // 2km = 0 points
  const schoolScore = Math.max(0, 10 - (schoolDist / 1)); // 1km = 0 points
  
  // Weighted average
  const weights = {
    highway: 0.15,
    metro: 0.25,
    airport: 0.10,
    itPark: 0.20,
    hospital: 0.15,
    school: 0.15
  };
  
  return (
    highwayScore * weights.highway +
    metroScore * weights.metro +
    airportScore * weights.airport +
    itParkScore * weights.itPark +
    hospitalScore * weights.hospital +
    schoolScore * weights.school
  );
};

// Calculate transport score
export const calculateTransportScore = (
  highwayDist: number,
  metroDist: number | null,
  airportDist: number
): number => {
  const highwayScore = Math.max(0, 10 - (highwayDist / 2));
  const metroScore = metroDist ? Math.max(0, 10 - (metroDist / 1)) : 5;
  const airportScore = Math.max(0, 10 - (airportDist / 5));
  
  return (highwayScore * 0.4 + metroScore * 0.4 + airportScore * 0.2);
};

// Calculate amenities score
export const calculateAmenitiesScore = (
  hospitalDist: number,
  schoolDist: number
): number => {
  const hospitalScore = Math.max(0, 10 - (hospitalDist / 2));
  const schoolScore = Math.max(0, 10 - (schoolDist / 1));
  
  return (hospitalScore * 0.5 + schoolScore * 0.5);
};

// Calculate employment score
export const calculateEmploymentScore = (
  itParkDist: number | null,
  highwayDist: number
): number => {
  const itParkScore = itParkDist ? Math.max(0, 10 - (itParkDist / 3)) : 5;
  const highwayScore = Math.max(0, 10 - (highwayDist / 2));
  
  return (itParkScore * 0.7 + highwayScore * 0.3);
};

// Classify connectivity level
export const classifyConnectivityLevel = (score: number): 'excellent' | 'good' | 'average' | 'poor' => {
  if (score >= 8) return 'excellent';
  if (score >= 6) return 'good';
  if (score >= 4) return 'average';
  return 'poor';
};

// Get color for connectivity level
export const getConnectivityColor = (level: string): string => {
  switch (level) {
    case 'excellent': return '#10b981'; // green
    case 'good': return '#3b82f6'; // blue
    case 'average': return '#f59e0b'; // orange
    case 'poor': return '#ef4444'; // red
    default: return '#6b7280'; // gray
  }
};

// Get color for score
export const getScoreColor = (score: number): string => {
  if (score >= 8) return '#10b981';
  if (score >= 6) return '#3b82f6';
  if (score >= 4) return '#f59e0b';
  return '#ef4444';
};

// Format distance
export const formatDistance = (km: number): string => {
  if (km < 1) return `${Math.round(km * 1000)}m`;
  return `${km.toFixed(1)}km`;
};

// Main formatter
export const formatInfrastructureData = (data: InfrastructureData): InfrastructureData => {
  return {
    ...data,
    accessibilityScore: Math.round(data.accessibilityScore * 10) / 10,
    transportScore: Math.round(data.transportScore * 10) / 10,
    amenitiesScore: Math.round(data.amenitiesScore * 10) / 10,
    employmentScore: Math.round(data.employmentScore * 10) / 10
  };
};
