import jsPDF from 'jspdf';
import { Transaction } from '@/types/transaction';

export const generateInvoice = (transaction: Transaction, userName: string, userEmail: string) => {
  const doc = new jsPDF();
  const pageWidth = doc.internal.pageSize.getWidth();
  const pageHeight = doc.internal.pageSize.getHeight();
  const margin = 15;
  let yPos = margin + 5;
  
  // Outer Border
  doc.setDrawColor(0, 0, 0);
  doc.setLineWidth(1);
  doc.rect(margin, margin, pageWidth - 2 * margin, pageHeight - 2 * margin);
  
  // Logos
  const nalLogo = new Image();
  const alstonairLogo = new Image();
  nalLogo.src = '/favicon.png';
  alstonairLogo.src = '/Black.png';
  
  try {
    doc.addImage(nalLogo, 'PNG', margin + 2, margin + 2, 20, 20);
    doc.addImage(alstonairLogo, 'PNG', pageWidth - margin - 45, margin + 2, 43, 15);
  } catch (e) {
    console.error('Error adding logos:', e);
  }
  
  yPos = margin + 25;
  
  // Header Section
  doc.setTextColor(0, 0, 0);
  doc.setFontSize(24);
  doc.setFont('helvetica', 'bold');
  doc.text('PAYMENT INVOICE', pageWidth / 2, yPos + 8, { align: 'center' });
  
  doc.setFontSize(10);
  doc.setFont('helvetica', 'normal');
  doc.text('New Age Land - Property Transaction Platform', pageWidth / 2, yPos + 16, { align: 'center' });
  
  yPos += 25;
  
  // Invoice Info Section
  doc.setTextColor(0, 0, 0);
  doc.setFontSize(10);
  doc.setFont('helvetica', 'bold');
  doc.text('Invoice Number:', margin + 5, yPos);
  doc.text('Invoice Date:', margin + 5, yPos + 8);
  
  doc.setFont('helvetica', 'normal');
  doc.text(transaction.transaction_number, margin + 50, yPos);
  doc.text(new Date(transaction.created_at).toLocaleDateString('en-IN', { 
    day: '2-digit', 
    month: 'short', 
    year: 'numeric' 
  }), margin + 50, yPos + 8);
  
  yPos += 20;
  
  // Bill To Section
  doc.setFont('helvetica', 'bold');
  doc.text('BILL TO:', margin + 5, yPos);
  yPos += 8;
  doc.setFont('helvetica', 'normal');
  doc.text(userName, margin + 5, yPos);
  yPos += 7;
  doc.setFontSize(9);
  doc.text(userEmail, margin + 5, yPos);
  
  yPos += 15;
  
  // Table Header
  doc.setFontSize(10);
  doc.setFillColor(230, 230, 230);
  doc.rect(margin + 5, yPos, pageWidth - 2 * margin - 10, 10, 'F');
  
  doc.setFont('helvetica', 'bold');
  doc.text('DESCRIPTION', margin + 10, yPos + 7);
  doc.text('AMOUNT', pageWidth - margin - 40, yPos + 7);
  
  yPos += 15;
  
  // Table Rows
  doc.setFont('helvetica', 'normal');
  
  // Line items
  doc.text(transaction.transaction_type.replace('_', ' ').toUpperCase(), margin + 10, yPos);
  doc.text('Rs. ' + transaction.amount.toFixed(2), pageWidth - margin - 40, yPos, { align: 'right' });
  
  yPos += 10;
  doc.text('GST (18%)', margin + 10, yPos);
  doc.text('Rs. ' + transaction.tax_amount.toFixed(2), pageWidth - margin - 40, yPos, { align: 'right' });
  
  yPos += 10;
  doc.text('Processing Fee (2%)', margin + 10, yPos);
  doc.text('Rs. ' + transaction.processing_fee.toFixed(2), pageWidth - margin - 40, yPos, { align: 'right' });
  
  // Total Line
  yPos += 8;
  doc.setDrawColor(0, 0, 0);
  doc.setLineWidth(0.5);
  doc.line(margin + 5, yPos, pageWidth - margin - 5, yPos);
  
  yPos += 10;
  doc.setFont('helvetica', 'bold');
  doc.setFontSize(12);
  doc.text('TOTAL AMOUNT', margin + 10, yPos);
  doc.text('Rs. ' + transaction.total_amount.toFixed(2), pageWidth - margin - 40, yPos, { align: 'right' });
  
  // Payment Details
  yPos += 20;
  doc.setFontSize(10);
  doc.setFont('helvetica', 'bold');
  doc.text('Payment Details:', margin + 10, yPos);
  
  doc.setFont('helvetica', 'normal');
  yPos += 8;
  doc.text('Payment Method: ' + transaction.payment_method.replace('_', ' ').toUpperCase(), margin + 10, yPos);
  
  yPos += 7;
  doc.setFontSize(8);
  doc.text('Transaction ID: ' + transaction.id, margin + 10, yPos);
  
  yPos += 7;
  doc.setFontSize(10);
  doc.setTextColor(0, 128, 0);
  doc.setFont('helvetica', 'bold');
  doc.text('Status: ' + transaction.transaction_status.toUpperCase(), margin + 10, yPos);
  
  // Thank You Note
  doc.setTextColor(100, 100, 100);
  doc.setFont('helvetica', 'italic');
  doc.setFontSize(9);
  doc.text('Thank you for your business!', pageWidth / 2, pageHeight - 35, { align: 'center' });
  doc.setFont('helvetica', 'normal');
  doc.setFontSize(8);
  doc.text('This is a computer-generated invoice and does not require a signature.', pageWidth / 2, pageHeight - 28, { align: 'center' });
  
  // Footer with Company Info
  doc.setFillColor(240, 240, 240);
  doc.rect(margin + 1, pageHeight - margin - 10, pageWidth - 2 * margin - 2, 9, 'F');
  
  doc.setTextColor(60, 60, 60);
  doc.setFont('helvetica', 'normal');
  doc.setFontSize(7);
  doc.text('© 2025 Alstonair Technologies Pvt. Ltd. | NAL India – Confidential Document', pageWidth / 2, pageHeight - margin - 4, { align: 'center' });
  
  // Save with timestamp to avoid caching
  doc.save(`Invoice_${transaction.transaction_number}_${Date.now()}.pdf`);
};
