import { Property } from '../data/properties';
import { ListingFormData } from '../types/listing';

export interface ScoreBreakdown {
  basicInfo: number;
  propertyDetails: number;
  location: number;
  media: number;
  amenities: number;
  legalInfo: number;
  total: number;
}

/**
 * Calculate listing score based on property completeness
 * Score ranges: 0-100
 * - 80-100: Fully completed listing
 * - 60-79: Good listing with minor missing info
 * - 40-59: Average listing with some missing info
 * - 20-39: Basic listing with significant missing info
 * - 0-19: Incomplete listing
 */
export function calculateListingScore(property: Property | ListingFormData): ScoreBreakdown {
  let basicInfo = 0;
  let propertyDetails = 0;
  let location = 0;
  let media = 0;
  let amenities = 0;
  let legalInfo = 0;

  // Basic Info (20 points max)
  if (property.title && property.title.length > 10) basicInfo += 5;
  if (property.description && property.description.length > 50) basicInfo += 5;
  if (property.price || ('salePrice' in property && property.salePrice)) basicInfo += 5;
  if ('beds' in property && property.beds) basicInfo += 2.5;
  if ('baths' in property && property.baths) basicInfo += 2.5;

  // Property Details (25 points max)
  if ('area' in property && property.area) propertyDetails += 5;
  if ('specifications' in property && property.specifications) {
    const specs = property.specifications;
    if (specs.propertyType) propertyDetails += 3;
    if (specs.facing) propertyDetails += 2;
    if (specs.floor && specs.totalFloors) propertyDetails += 3;
    if (specs.age) propertyDetails += 2;
    if (specs.furnishing) propertyDetails += 2;
    if (specs.parking) propertyDetails += 3;
  } else {
    // For ListingFormData
    if ('propertyType' in property && property.propertyType) propertyDetails += 3;
    if ('facing' in property && property.facing) propertyDetails += 2;
    if ('floorNumber' in property && property.floorNumber && 'totalFloors' in property && property.totalFloors) propertyDetails += 3;
    if ('propertyAge' in property && property.propertyAge) propertyDetails += 2;
    if ('furnishing' in property && property.furnishing) propertyDetails += 2;
    if ('parking' in property && property.parking) propertyDetails += 3;
  }
  if ('builtUpArea' in property && property.builtUpArea) propertyDetails += 2;
  if ('carpetArea' in property && property.carpetArea) propertyDetails += 2;
  if ('bedrooms' in property && property.bedrooms) propertyDetails += 1;
  if ('bathrooms' in property && property.bathrooms) propertyDetails += 1;

  // Location (15 points max)
  if ('location' in property && property.location) location += 5;
  if ('city' in property && property.city) location += 3;
  if ('locality' in property && property.locality) location += 3;
  if ('address' in property && property.address) location += 2;
  if (('latitude' in property && property.latitude) || ('coordinates' in property && property.coordinates)) location += 2;

  // Media (20 points max)
  if ('images' in property && property.images && property.images.length > 0) {
    if (property.images.length >= 5) media += 10;
    else if (property.images.length >= 3) media += 7;
    else if (property.images.length >= 1) media += 4;
  }
  if ('image' in property && property.image) media += 4; // Primary image
  if ('videos' in property && property.videos && property.videos.length > 0) media += 3;
  if ('virtualTourUrl' in property && property.virtualTourUrl) media += 3;

  // Amenities & Features (10 points max)
  if ('amenities' in property && property.amenities && property.amenities.length > 0) {
    if (property.amenities.length >= 8) amenities += 6;
    else if (property.amenities.length >= 5) amenities += 4;
    else if (property.amenities.length >= 2) amenities += 2;
  }
  if ('features' in property && property.features && property.features.length > 0) amenities += 2;
  if ('highlights' in property && property.highlights && property.highlights.length > 0) amenities += 2;

  // Legal Info & Documentation (10 points max)
  if ('legalInfo' in property && property.legalInfo) {
    const legal = property.legalInfo;
    if (legal.rera) legalInfo += 4;
    if (legal.approvals && legal.approvals.length > 0) legalInfo += 3;
    if (legal.ownership) legalInfo += 2;
  } else {
    // For ListingFormData
    if ('reraNumber' in property && property.reraNumber) legalInfo += 4;
    if ('documents' in property && property.documents && property.documents.length > 0) legalInfo += 3;
    if ('legalDocuments' in property && property.legalDocuments && property.legalDocuments.length > 0) legalInfo += 2;
  }
  if ('kycVerified' in property && property.kycVerified) legalInfo += 1;

  const total = Math.min(100, Math.round(basicInfo + propertyDetails + location + media + amenities + legalInfo));

  return {
    basicInfo: Math.round(basicInfo),
    propertyDetails: Math.round(propertyDetails),
    location: Math.round(location),
    media: Math.round(media),
    amenities: Math.round(amenities),
    legalInfo: Math.round(legalInfo),
    total
  };
}

/**
 * Get score color based on total score
 */
export function getScoreColor(score: number): string {
  if (score >= 80) return 'text-green-600';
  if (score >= 60) return 'text-blue-600';
  if (score >= 40) return 'text-yellow-600';
  if (score >= 20) return 'text-orange-600';
  return 'text-red-600';
}

/**
 * Get score background color based on total score
 */
export function getScoreBgColor(score: number): string {
  if (score >= 80) return 'bg-green-100';
  if (score >= 60) return 'bg-blue-100';
  if (score >= 40) return 'bg-yellow-100';
  if (score >= 20) return 'bg-orange-100';
  return 'bg-red-100';
}

/**
 * Get score label based on total score
 */
export function getScoreLabel(score: number): string {
  if (score >= 80) return 'Excellent';
  if (score >= 60) return 'Good';
  if (score >= 40) return 'Average';
  if (score >= 20) return 'Basic';
  return 'Incomplete';
}