// Enhanced caching utility for LocalityIntelligence
interface CacheEntry {
  data: any;
  timestamp: number;
  accessCount: number;
  lastAccessed: number;
  size: number;
}

interface CacheStats {
  totalEntries: number;
  totalSize: number;
  hitRate: number;
  mostAccessed: string[];
  oldestEntry: string;
  newestEntry: string;
}

class LocalityCache {
  private cache = new Map<string, CacheEntry>();
  private hits = 0;
  private misses = 0;
  private readonly maxSize = 50;
  private readonly maxAge = 24 * 60 * 60 * 1000; // 24 hours

  constructor() {
    this.loadFromStorage();
    this.startCleanupInterval();
  }

  get(key: string, maxAge?: number): any | null {
    const entry = this.cache.get(key);
    const age = maxAge || this.maxAge;
    
    if (!entry) {
      this.misses++;
      return null;
    }

    if (Date.now() - entry.timestamp > age) {
      this.cache.delete(key);
      this.misses++;
      return null;
    }

    entry.accessCount++;
    entry.lastAccessed = Date.now();
    this.hits++;
    
    console.log(`📦 Cache HIT for: ${key}`);
    return entry.data;
  }

  set(key: string, data: any): void {
    const size = JSON.stringify(data).length;
    
    if (this.cache.size >= this.maxSize) {
      this.evictOldest();
    }

    const entry: CacheEntry = {
      data,
      timestamp: Date.now(),
      accessCount: 1,
      lastAccessed: Date.now(),
      size
    };

    this.cache.set(key, entry);
    this.saveToStorage();
    
    console.log(`💾 Cached: ${key} (${(size / 1024).toFixed(2)}KB)`);
  }

  delete(key: string): boolean {
    const deleted = this.cache.delete(key);
    if (deleted) {
      this.saveToStorage();
      console.log(`🗑️ Deleted cache entry: ${key}`);
    }
    return deleted;
  }

  clear(): void {
    this.cache.clear();
    this.hits = 0;
    this.misses = 0;
    localStorage.removeItem('locality_cache');
    console.log('🧹 Cache cleared completely');
  }

  getStats(): CacheStats {
    const entries = Array.from(this.cache.entries());
    const totalSize = entries.reduce((sum, [, entry]) => sum + entry.size, 0);
    const hitRate = this.hits + this.misses > 0 ? (this.hits / (this.hits + this.misses)) * 100 : 0;
    
    const sortedByAccess = entries
      .sort(([, a], [, b]) => b.accessCount - a.accessCount)
      .slice(0, 5)
      .map(([key]) => key);

    const sortedByTime = entries.sort(([, a], [, b]) => a.timestamp - b.timestamp);
    const oldestEntry = sortedByTime[0]?.[0] || '';
    const newestEntry = sortedByTime[sortedByTime.length - 1]?.[0] || '';

    return {
      totalEntries: this.cache.size,
      totalSize,
      hitRate: Number(hitRate.toFixed(2)),
      mostAccessed: sortedByAccess,
      oldestEntry,
      newestEntry
    };
  }

  getKeys(): string[] {
    return Array.from(this.cache.keys());
  }

  has(key: string, maxAge?: number): boolean {
    const entry = this.cache.get(key);
    if (!entry) return false;
    
    const age = maxAge || this.maxAge;
    return Date.now() - entry.timestamp <= age;
  }

  private evictOldest(): void {
    let oldestKey = '';
    let oldestTime = Date.now();
    
    for (const [key, entry] of this.cache.entries()) {
      if (entry.lastAccessed < oldestTime) {
        oldestTime = entry.lastAccessed;
        oldestKey = key;
      }
    }
    
    if (oldestKey) {
      this.cache.delete(oldestKey);
      console.log(`🔄 Evicted oldest cache entry: ${oldestKey}`);
    }
  }

  private loadFromStorage(): void {
    try {
      const stored = localStorage.getItem('locality_cache');
      if (stored) {
        const parsed = JSON.parse(stored);
        
        if (parsed.entries) {
          this.cache = new Map(parsed.entries);
        }
        
        this.hits = parsed.hits || 0;
        this.misses = parsed.misses || 0;
        
        console.log(`📂 Loaded ${this.cache.size} cache entries from storage`);
      }
    } catch (error) {
      console.warn('⚠️ Failed to load cache from storage:', error);
      this.cache.clear();
    }
  }

  private saveToStorage(): void {
    try {
      const cacheData = {
        entries: Array.from(this.cache.entries()),
        hits: this.hits,
        misses: this.misses,
        lastSaved: Date.now()
      };
      
      localStorage.setItem('locality_cache', JSON.stringify(cacheData));
    } catch (error) {
      console.warn('⚠️ Failed to save cache to storage:', error);
    }
  }

  private startCleanupInterval(): void {
    setInterval(() => {
      this.cleanupExpired();
    }, 5 * 60 * 1000); // Cleanup every 5 minutes
  }

  private cleanupExpired(): void {
    const now = Date.now();
    let cleanedCount = 0;
    
    for (const [key, entry] of this.cache.entries()) {
      if (now - entry.timestamp > this.maxAge) {
        this.cache.delete(key);
        cleanedCount++;
      }
    }
    
    if (cleanedCount > 0) {
      this.saveToStorage();
      console.log(`🧹 Cleaned up ${cleanedCount} expired cache entries`);
    }
  }
}

export const localityCache = new LocalityCache();

export const cacheUtils = {
  getLocality: (location: string, reportType: string = 'comprehensive') => {
    const key = `locality_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}_${reportType}`;
    return localityCache.get(key);
  },

  setLocality: (location: string, data: any, reportType: string = 'comprehensive') => {
    const key = `locality_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}_${reportType}`;
    localityCache.set(key, data);
  },

  getLiveStats: () => {
    return localityCache.get('live_stats', 5 * 60 * 1000); // 5 minutes
  },

  setLiveStats: (data: any) => {
    localityCache.set('live_stats', data);
  },

  getSentiment: (location: string) => {
    const key = `sentiment_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}`;
    return localityCache.get(key);
  },

  setSentiment: (location: string, data: any) => {
    const key = `sentiment_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}`;
    localityCache.set(key, data);
  },

  getMarketTrends: (location: string) => {
    const key = `trends_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}`;
    return localityCache.get(key, 60 * 60 * 1000); // 1 hour
  },

  setMarketTrends: (location: string, data: any) => {
    const key = `trends_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}`;
    localityCache.set(key, data);
  },

  clearLocalityCache: () => {
    const keys = localityCache.getKeys();
    const localityKeys = keys.filter(key => 
      key.startsWith('locality_') || 
      key.startsWith('sentiment_') || 
      key.startsWith('trends_')
    );
    
    localityKeys.forEach(key => localityCache.delete(key));
    console.log(`🧹 Cleared ${localityKeys.length} locality cache entries`);
  }
};

export default localityCache;
