/**
 * High-performance marker clustering - NO LAG BULLSHIT
 */

import { Property, HeatmapPoint } from '../components/heatmap/types';

interface ClusterPoint {
  lat: number;
  lng: number;
  count: number;
  properties: Property[];
  bounds: {
    north: number;
    south: number;
    east: number;
    west: number;
  };
}

interface ViewportBounds {
  north: number;
  south: number;
  east: number;
  west: number;
}

export class MarkerClustering {
  private static readonly MAX_ZOOM_CLUSTER = 15;
  private static readonly MIN_CLUSTER_SIZE = 2;

  /**
   * Cluster markers based on zoom level and viewport
   */
  static clusterMarkers(
    points: HeatmapPoint[], 
    zoomLevel: number, 
    viewport: ViewportBounds
  ): ClusterPoint[] {
    // No clustering at high zoom levels
    if (zoomLevel >= this.MAX_ZOOM_CLUSTER) {
      return points.map(point => ({
        lat: point.lat,
        lng: point.lng,
        count: 1,
        properties: [],
        bounds: {
          north: point.lat,
          south: point.lat,
          east: point.lng,
          west: point.lng
        }
      }));
    }

    // Filter points in viewport first
    const visiblePoints = points.filter(point => 
      point.lat >= viewport.south && point.lat <= viewport.north &&
      point.lng >= viewport.west && point.lng <= viewport.east
    );

    if (visiblePoints.length === 0) return [];

    // Dynamic cluster radius based on zoom
    const clusterRadius = this.getClusterRadius(zoomLevel);
    const clusters: ClusterPoint[] = [];
    const processed = new Set<number>();

    visiblePoints.forEach((point, index) => {
      if (processed.has(index)) return;

      const cluster: ClusterPoint = {
        lat: point.lat,
        lng: point.lng,
        count: 1,
        properties: [],
        bounds: {
          north: point.lat,
          south: point.lat,
          east: point.lng,
          west: point.lng
        }
      };

      processed.add(index);

      // Find nearby points to cluster
      visiblePoints.forEach((otherPoint, otherIndex) => {
        if (processed.has(otherIndex)) return;

        const distance = this.calculateDistance(
          point.lat, point.lng,
          otherPoint.lat, otherPoint.lng
        );

        if (distance <= clusterRadius) {
          cluster.count++;
          cluster.lat = (cluster.lat * (cluster.count - 1) + otherPoint.lat) / cluster.count;
          cluster.lng = (cluster.lng * (cluster.count - 1) + otherPoint.lng) / cluster.count;
          
          // Update bounds
          cluster.bounds.north = Math.max(cluster.bounds.north, otherPoint.lat);
          cluster.bounds.south = Math.min(cluster.bounds.south, otherPoint.lat);
          cluster.bounds.east = Math.max(cluster.bounds.east, otherPoint.lng);
          cluster.bounds.west = Math.min(cluster.bounds.west, otherPoint.lng);
          
          processed.add(otherIndex);
        }
      });

      clusters.push(cluster);
    });

    return clusters;
  }

  /**
   * Viewport-based loading for performance
   */
  static getViewportBounds(map: google.maps.Map): ViewportBounds {
    const bounds = map.getBounds();
    if (!bounds) {
      return { north: 90, south: -90, east: 180, west: -180 };
    }

    const ne = bounds.getNorthEast();
    const sw = bounds.getSouthWest();

    return {
      north: ne.lat(),
      south: sw.lat(),
      east: ne.lng(),
      west: sw.lng()
    };
  }

  /**
   * Memory-efficient marker management
   */
  static createOptimizedMarkers(
    clusters: ClusterPoint[],
    map: google.maps.Map,
    onClick: (cluster: ClusterPoint) => void
  ): google.maps.Circle[] {
    return clusters.map(cluster => {
      const circle = new google.maps.Circle({
        strokeColor: this.getClusterColor(cluster.count),
        strokeOpacity: 0.8,
        strokeWeight: 2,
        fillColor: this.getClusterColor(cluster.count),
        fillOpacity: 0.6,
        map: map,
        center: { lat: cluster.lat, lng: cluster.lng },
        radius: this.getClusterRadius(map.getZoom() || 12) * 100,
        clickable: true
      });

      circle.addListener('click', () => onClick(cluster));
      return circle;
    });
  }

  /**
   * Cleanup markers to prevent memory leaks
   */
  static cleanupMarkers(markers: google.maps.Circle[]): void {
    markers.forEach(marker => {
      google.maps.event.clearInstanceListeners(marker);
      marker.setMap(null);
    });
  }

  /**
   * Get cluster radius based on zoom level
   */
  private static getClusterRadius(zoomLevel: number): number {
    if (zoomLevel >= 15) return 0.001; // 100m
    if (zoomLevel >= 12) return 0.005; // 500m
    if (zoomLevel >= 10) return 0.01;  // 1km
    return 0.02; // 2km
  }

  /**
   * Get cluster color based on density
   */
  private static getClusterColor(count: number): string {
    if (count >= 10) return '#FF4444'; // High density - Red
    if (count >= 5) return '#FF8800';  // Medium density - Orange
    if (count >= 2) return '#FFDD00';  // Low density - Yellow
    return '#44AA44'; // Single - Green
  }

  /**
   * Fast distance calculation
   */
  private static calculateDistance(lat1: number, lng1: number, lat2: number, lng2: number): number {
    const dLat = lat2 - lat1;
    const dLng = lng2 - lng1;
    return Math.sqrt(dLat * dLat + dLng * dLng);
  }
}

export default MarkerClustering;