import { MarketPositioning } from '../types/marketPositioning';

interface PropertyData {
  id: number;
  sale_price?: number;
  monthly_rent?: number;
  built_up_area?: number;
  property_type: string;
  locality: string;
  city: string;
  bedrooms?: number;
  market_positioning?: MarketPositioning | null;
}

export function enrichPropertiesWithMarketData<T extends PropertyData>(properties: T[]): T[] {
  return properties.map(property => {
    const price = property.sale_price || property.monthly_rent || 0;
    const area = property.built_up_area || 0;
    
    if (price === 0 || area === 0) {
      return {
        ...property,
        market_positioning: null
      };
    }

    const pricePerSqft = price / area;
    
    // Calculate locality average
    const localityProperties = properties.filter(p => 
      p.locality === property.locality && 
      p.property_type === property.property_type &&
      (p.sale_price || p.monthly_rent || 0) > 0 &&
      (p.built_up_area || 0) > 0
    );
    
    const localityAvgPricePerSqft = localityProperties.length > 0
      ? localityProperties.reduce((sum, p) => {
          const pPrice = p.sale_price || p.monthly_rent || 0;
          const pArea = p.built_up_area || 0;
          return sum + (pPrice / pArea);
        }, 0) / localityProperties.length
      : pricePerSqft;

    const deviation = ((pricePerSqft - localityAvgPricePerSqft) / localityAvgPricePerSqft) * 100;
    
    let marketPositionTag: 'HIGH_VALUE' | 'SMART_BUY' | 'BALANCED' | 'OVERPRICED' | 'EXTREME_OVERPRICED' | 'UNKNOWN';
    
    if (deviation < -15) {
      marketPositionTag = 'HIGH_VALUE';
    } else if (deviation < -5) {
      marketPositionTag = 'SMART_BUY';
    } else if (deviation <= 10) {
      marketPositionTag = 'BALANCED';
    } else if (deviation <= 25) {
      marketPositionTag = 'OVERPRICED';
    } else {
      marketPositionTag = 'EXTREME_OVERPRICED';
    }

    return {
      ...property,
      market_positioning: {
        market_position_tag: marketPositionTag,
        price_per_sqft: pricePerSqft,
        locality_avg_price_per_sqft: localityAvgPricePerSqft,
        deviation_percentage: deviation
      }
    };
  });
}
