// Media API integration for photos and videos
export interface PropertyMedia {
  id: number;
  property_id: number;
  media_type: 'photo' | 'video';
  file_url: string;
  thumbnail_url?: string;
  filename: string;
  file_size?: number;
  duration?: number; // for videos
  user_id: string;
  created_at: string;
}

export interface MediaResponse {
  media: PropertyMedia[];
  total_count: number;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem('token');
  return token ? { 'Authorization': `Bearer ${token}` } : {};
};

export interface MediaGalleryItem {
  id: number;
  media_type: 'image' | 'video';
  file_url: string;
  thumbnail_url?: string;
  alt_text?: string;
  category?: string;
  user_id: string;
  created_at: string;
}

export interface GalleryResponse {
  items: MediaGalleryItem[];
  total_count: number;
}

export const mediaApi = {
  // Upload media (photos/videos)
  uploadMedia: async (
    propertyId: number,
    file: File,
    mediaType: 'photo' | 'video'
  ): Promise<PropertyMedia> => {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('media_type', mediaType);

    const response = await fetch(`/api/property-media/upload?property_id=${propertyId}`, {
      method: 'POST',
      headers: getAuthHeaders(),
      body: formData
    });

    if (!response.ok) {
      const error = await response.text();
      throw new Error(`Upload failed: ${response.status} ${error}`);
    }

    return response.json();
  },

  // Get media for property
  getPropertyMedia: async (propertyId: number): Promise<MediaResponse> => {
    try {
      const response = await fetch(`/api/property-media/${propertyId}`, {
        headers: getAuthHeaders()
      });

      if (response.ok) {
        return response.json();
      }
    } catch (error) {
      console.log('Media API not available, using mock data');
    }

    // Mock data fallback
    return {
      media: [
        {
          id: propertyId * 100 + 1,
          property_id: propertyId,
          media_type: 'photo',
          file_url: `https://nalpropertysa.blob.core.windows.net/media/${propertyId}/photo1.jpg`,
          thumbnail_url: `https://nalpropertysa.blob.core.windows.net/media/${propertyId}/thumb1.jpg`,
          filename: 'exterior_view.jpg',
          file_size: 2048576,
          user_id: `user_${propertyId}`,
          created_at: new Date().toISOString()
        },
        {
          id: propertyId * 100 + 2,
          property_id: propertyId,
          media_type: 'video',
          file_url: `https://nalpropertysa.blob.core.windows.net/media/${propertyId}/tour.mp4`,
          thumbnail_url: `https://nalpropertysa.blob.core.windows.net/media/${propertyId}/tour_thumb.jpg`,
          filename: 'property_tour.mp4',
          file_size: 15728640,
          duration: 180,
          user_id: `user_${propertyId}`,
          created_at: new Date().toISOString()
        }
      ],
      total_count: 2
    };
  },

  // Generate download URL for media
  getDownloadUrl: async (mediaId: number): Promise<{ download_url: string; expires_in_hours: number }> => {
    try {
      const response = await fetch(`/api/property-media/${mediaId}/download`, {
        headers: getAuthHeaders()
      });

      if (response.ok) {
        return response.json();
      }
    } catch (error) {
      console.log('Download API not available, using direct URL');
    }

    return {
      download_url: `https://nalpropertysa.blob.core.windows.net/media/mock/media_${mediaId}`,
      expires_in_hours: 24
    };
  },

  // Delete media
  deleteMedia: async (mediaId: number): Promise<void> => {
    const response = await fetch(`/api/property-media/${mediaId}`, {
      method: 'DELETE',
      headers: getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Delete failed');
    }
  },

  // Get user gallery
  getUserGallery: async (userId: string, mediaType: 'image' | 'video'): Promise<GalleryResponse> => {
    try {
      const response = await fetch(`/api/user-gallery/${userId}?media_type=${mediaType}`, {
        headers: getAuthHeaders()
      });

      if (response.ok) {
        return response.json();
      }
    } catch (error) {
      console.log('User gallery API not available, using mock data');
    }

    // Mock data fallback
    return {
      items: [],
      total_count: 0
    };
  },

  // Upload to gallery
  uploadToGallery: async (file: File, userId: string, category?: string): Promise<MediaGalleryItem> => {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('user_id', userId);
    if (category) formData.append('category', category);

    try {
      const response = await fetch('/api/user-gallery/upload', {
        method: 'POST',
        headers: getAuthHeaders(),
        body: formData
      });

      if (response.ok) {
        return response.json();
      }
    } catch (error) {
      console.log('Gallery upload API not available, creating mock item');
    }

    // Mock upload for development
    const mockId = Date.now() + Math.random();
    const fileUrl = URL.createObjectURL(file);
    
    return {
      id: mockId,
      media_type: file.type.startsWith('video/') ? 'video' : 'image',
      file_url: fileUrl,
      thumbnail_url: file.type.startsWith('image/') ? fileUrl : undefined,
      alt_text: file.name,
      category: category || 'other',
      user_id: userId,
      created_at: new Date().toISOString()
    };
  }
};