// NRI Dashboard Utility Functions and Types

export interface NRIRequest {
  id: string;
  name: string;
  email: string;
  phone: string;
  country: string;
  category: string;
  status: 'New' | 'In Progress' | 'Resolved' | 'Closed';
  priority: 'Low' | 'Medium' | 'High';
  officer: string;
  createdDate: string;
  resolutionTime: string;
  description: string;
}

export interface DashboardStats {
  totalRequests: number;
  resolvedRequests: number;
  pendingRequests: number;
  avgResolutionTime: string;
  monthlyGrowth: string;
  satisfactionRate: string;
}

export interface ChartData {
  categoryWise: Array<{
    category: string;
    count: number;
    percentage: number;
    color: string;
  }>;
  statusWise: Array<{
    name: string;
    value: number;
    color: string;
  }>;
  monthlyTrend: Array<{
    month: string;
    requests: number;
    resolved: number;
    pending: number;
  }>;
}

// Constants
export const COUNTRIES = [
  'All', 'USA', 'UK', 'Canada', 'Australia', 'UAE', 'Singapore', 'Germany', 'Other'
];

export const CATEGORIES = [
  'All', 'Property', 'Legal', 'Documentation', 'Banking', 'Other'
];

export const STATUSES = [
  'All', 'New', 'In Progress', 'Resolved', 'Closed'
];

export const PRIORITIES = [
  'All', 'Low', 'Medium', 'High'
];

export const DASHBOARD_COLORS = {
  primary: '#0056D2',
  success: '#10B981',
  warning: '#F59E0B',
  danger: '#EF4444',
  info: '#3B82F6'
};

// Utility Functions
export const getStatusBadgeClass = (status: string): string => {
  switch (status) {
    case 'New':
      return 'bg-blue-100 text-blue-800 border-blue-200';
    case 'In Progress':
      return 'bg-orange-100 text-orange-800 border-orange-200';
    case 'Resolved':
      return 'bg-green-100 text-green-800 border-green-200';
    case 'Closed':
      return 'bg-gray-100 text-gray-800 border-gray-200';
    default:
      return 'bg-gray-100 text-gray-800 border-gray-200';
  }
};

export const getPriorityBadgeClass = (priority: string): string => {
  switch (priority) {
    case 'High':
      return 'bg-red-100 text-red-800 border-red-200';
    case 'Medium':
      return 'bg-yellow-100 text-yellow-800 border-yellow-200';
    case 'Low':
      return 'bg-green-100 text-green-800 border-green-200';
    default:
      return 'bg-gray-100 text-gray-800 border-gray-200';
  }
};

export const formatDate = (dateString: string): string => {
  const date = new Date(dateString);
  return date.toLocaleDateString('en-US', {
    year: 'numeric',
    month: 'short',
    day: 'numeric'
  });
};

export const exportToCSV = (data: NRIRequest[], filename: string): void => {
  const headers = [
    'Request ID',
    'Name',
    'Email',
    'Phone',
    'Country',
    'Category',
    'Status',
    'Priority',
    'Officer',
    'Created Date',
    'Resolution Time',
    'Description'
  ];

  const csvContent = [
    headers.join(','),
    ...data.map(row => [
      row.id,
      row.name,
      row.email,
      row.phone,
      row.country,
      row.category,
      row.status,
      row.priority,
      row.officer,
      row.createdDate,
      row.resolutionTime,
      `"${row.description.replace(/"/g, '""')}"`
    ].join(','))
  ].join('\n');

  const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
  const link = document.createElement('a');
  const url = URL.createObjectURL(blob);
  link.setAttribute('href', url);
  link.setAttribute('download', filename);
  link.style.visibility = 'hidden';
  document.body.appendChild(link);
  link.click();
  document.body.removeChild(link);
};