// Mock PDF generation utility for Custom Property Dossier
// This will be replaced with actual PDF generation in Phase 3

export interface PropertyData {
  id: string;
  surveyNo: string;
  address: string;
  type: string;
  area: string;
  owner: string;
}

export interface DossierSection {
  id: string;
  title: string;
  description: string;
  included: boolean;
}

// Mock data for different sections
const mockSectionData = {
  basic: {
    propertyId: 'PROP-2024-001',
    type: 'Residential Apartment',
    address: '123 Marine Drive, Bandra West, Mumbai - 400050',
    area: '1200 sq ft',
    builtUpArea: '1400 sq ft',
    usage: 'Residential',
    floors: '12th Floor of 25 floors',
    facing: 'West facing with sea view'
  },
  ownership: {
    currentOwner: 'Rajesh Kumar',
    ownershipType: 'Individual',
    acquisitionDate: '2020-03-15',
    previousOwner: 'Mumbai Housing Development',
    ownershipHistory: [
      { owner: 'Mumbai Housing Development', period: '2018-2020' },
      { owner: 'Rajesh Kumar', period: '2020-Present' }
    ]
  },
  legal: {
    saleDeadNo: 'SD/2020/12345',
    registrationNo: 'REG/MUM/2020/6789',
    registrationDate: '2020-03-20',
    encumbranceStatus: 'Clear',
    litigationStatus: 'No pending litigation',
    approvals: ['Building Plan Approval', 'Occupancy Certificate', 'Fire NOC']
  },
  tax: {
    propertyTaxId: 'PT/MUM/2024/001',
    lastPaidAmount: '₹45,000',
    lastPaidDate: '2024-01-15',
    taxStatus: 'Paid',
    electricityConnection: 'Active - Consumer No: ELE123456',
    waterConnection: 'Active - Consumer No: WAT789012'
  },
  location: {
    coordinates: { lat: 19.0596, lng: 72.8295 },
    landmarks: ['Bandra-Worli Sea Link - 1.5 km', 'Linking Road - 2 km', 'Bandra Station - 1.8 km'],
    zoning: 'Residential Zone R1',
    roadAccess: 'Direct road access from Marine Drive',
    connectivity: 'Excellent connectivity to Western Express Highway'
  },
  documents: {
    available: [
      'Sale Deed Copy',
      'Property Tax Receipt',
      'Electricity Bill',
      'Society NOC',
      'Building Plan'
    ],
    pending: [
      'Latest Encumbrance Certificate',
      'Property Valuation Report'
    ]
  }
};

export const generatePropertyDossierPDF = async (
  property: PropertyData,
  selectedSections: DossierSection[]
): Promise<string> => {
  // Simulate PDF generation process
  await new Promise(resolve => setTimeout(resolve, 2000));
  
  // In real implementation, this would:
  // 1. Collect data for selected sections
  // 2. Generate PDF using a library like jsPDF or PDFKit
  // 3. Return the PDF blob URL or base64 string
  
  console.log('Generating PDF for property:', property.id);
  console.log('Selected sections:', selectedSections.map(s => s.title));
  
  // Mock PDF content generation
  const pdfContent = {
    title: `Property Dossier - ${property.id}`,
    generatedOn: new Date().toISOString(),
    property: property,
    sections: selectedSections.map(section => ({
      ...section,
      data: mockSectionData[section.id as keyof typeof mockSectionData] || {}
    }))
  };
  
  // Return mock PDF URL (in real implementation, this would be actual PDF)
  return `data:application/pdf;base64,${btoa(JSON.stringify(pdfContent))}`;
};

export const downloadPDF = (pdfUrl: string, filename: string) => {
  // In real implementation, this would download actual PDF
  // For now, we'll just show the mock data
  
  const link = document.createElement('a');
  link.href = pdfUrl;
  link.download = filename;
  document.body.appendChild(link);
  link.click();
  document.body.removeChild(link);
  
  // Show mock download success
  alert(`PDF "${filename}" would be downloaded here.\n\nIn Phase 3, this will generate and download a real PDF with:\n- Property details\n- Selected sections data\n- Professional formatting\n- Branded template`);
};

// Utility to get section data
export const getSectionData = (sectionId: string) => {
  return mockSectionData[sectionId as keyof typeof mockSectionData] || {};
};

// Utility to estimate PDF size based on selected sections
export const estimatePDFSize = (selectedSections: DossierSection[]): string => {
  const baseSize = 0.5; // MB
  const sectionSize = selectedSections.length * 0.3; // MB per section
  const totalSize = baseSize + sectionSize;
  
  return `${totalSize.toFixed(1)} MB`;
};