import { uploadPropertyImages, uploadPropertyVideos } from '../api/mediaUpload';
import { toast } from 'sonner';

// Helper function to convert base64 to File
const base64ToFile = (base64Data: string, fileName: string, fileType: string): File => {
  const byteCharacters = atob(base64Data.split(',')[1]);
  const byteNumbers = new Array(byteCharacters.length);
  for (let i = 0; i < byteCharacters.length; i++) {
    byteNumbers[i] = byteCharacters.charCodeAt(i);
  }
  const byteArray = new Uint8Array(byteNumbers);
  return new File([byteArray], fileName, { type: fileType });
};

export const uploadPendingMediaFiles = async (propertyId: string) => {
  try {
    const pendingFiles = (window as any).pendingMediaFiles || [];
    
    if (pendingFiles.length === 0) {
      return { success: true, message: 'No pending files to upload' };
    }

    console.log(`📤 Uploading ${pendingFiles.length} pending files simultaneously for property ${propertyId}`);

    // Separate images and videos (files are already File objects)
    const imageFiles = pendingFiles.filter((item: any) => item.type === 'image').map((item: any) => item.file);
    const videoFiles = pendingFiles.filter((item: any) => item.type === 'video').map((item: any) => item.file);

    // Upload ALL files simultaneously
    const uploadPromises = [];
    
    if (imageFiles.length > 0) {
      uploadPromises.push(uploadPropertyImages(propertyId, imageFiles));
    }
    
    if (videoFiles.length > 0) {
      uploadPromises.push(uploadPropertyVideos(propertyId, videoFiles));
    }

    // Execute all uploads simultaneously
    const results = await Promise.all(uploadPromises);
    
    // Clear pending files
    delete (window as any).pendingMediaFiles;

    const successCount = results.filter(r => r.success).length;
    if (successCount > 0) {
      toast.success(`✅ All ${pendingFiles.length} files uploaded simultaneously!`);
      return { success: true, message: `${pendingFiles.length} files uploaded` };
    } else {
      toast.error('❌ Failed to upload media files');
      return { success: false, message: 'Upload failed' };
    }

  } catch (error) {
    console.error('Error uploading pending media files:', error);
    toast.error('❌ Error uploading media files');
    return { success: false, error: error instanceof Error ? error.message : 'Upload failed' };
  }
};