import { propertyIntentClassifier } from './propertyIntentClassifier';
import { kannadaResponses, furnishingDetails, negotiationText } from './kannadaResponses';
import { 
  translateKannadaToEnglish, 
  isKannada, 
  formatKannadaResponse,
  translateAmenitiesToKannada,
  formatAgeInKannada,
  formatNearbyInKannada
} from './translationHelper';

interface Property {
  id: number;
  owner_name: string;
  title: string;
  property_type: string;
  locality: string;
  city: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  carpet_area?: number;
  sale_price?: string;
  monthly_rent?: string;
  furnishing?: string;
  amenities?: string[];
  price_negotiable?: boolean;
  facing?: string;
  property_age?: string;
  floor_number?: string;
  total_floors?: string;
  nearby_places?: string[];
}

export class PropertyAssistant {
  private property: Property;
  private language: string;

  constructor(property: Property, language: string = 'en') {
    this.property = property;
    this.language = language;
  }

  getGreeting(): string {
    if (this.language === 'kn') {
      return kannadaResponses.GREETING;
    }
    
    const ownerName = this.property.owner_name || 'the owner';
    const propertyType = this.property.property_type || 'property';
    const locality = this.property.locality;
    const bedrooms = this.property.bedrooms;
    const price = this.formatPrice(this.property.sale_price || this.property.monthly_rent || '0');

    return `Hi! I'm representing ${ownerName} for this beautiful ${bedrooms ? `${bedrooms} bedroom ` : ''}${propertyType} in ${locality}. It's priced at ${price} and has been well maintained. What would you like to know about this property?`;
  }

  processQuery(question: string): string {
    return this.processQuestion(question);
  }

  processQuestion(question: string): string {
    let q = question.toLowerCase().trim();

    // If Kannada, translate to English for intent classification
    if (this.language === 'kn' && isKannada(q)) {
      q = translateKannadaToEnglish(q);
      console.log('Translated to English:', q);
    }

    // Use intent classifier for better understanding
    const { intent, confidence, entities } = propertyIntentClassifier.classify(q);

    let response = '';

    // Route based on classified intent
    switch (intent) {
      case 'GREETING':
        response = this.getGreetingResponse();
        break;
      case 'PRICE_QUERY':
        response = this.getPriceResponse(entities);
        break;
      case 'SIZE_QUERY':
        response = this.getSizeResponse(entities);
        break;
      case 'LOCATION_QUERY':
        response = this.getLocationResponse(entities);
        break;
      case 'AMENITIES_QUERY':
        response = this.getAmenitiesResponse(entities);
        break;
      case 'ROOMS_QUERY':
        response = this.getRoomsResponse(entities);
        break;
      case 'NEGOTIATION_QUERY':
        response = this.getNegotiationResponse();
        break;
      case 'CONDITION_QUERY':
        response = this.getConditionResponse();
        break;
      case 'VISIT_QUERY':
        response = this.getVisitResponse();
        break;
      case 'COMPARISON_QUERY':
        response = this.getComparisonResponse();
        break;
      case 'INVESTMENT_QUERY':
        response = this.getInvestmentResponse();
        break;
      case 'FURNISHING_QUERY':
        response = this.getFurnishingResponse();
        break;
      default:
        // Low confidence - ask for clarification
        if (confidence < 0.5) {
          response = this.getClarificationResponse();
        } else {
          response = this.getDefaultResponse();
        }
    }

    return response;
  }

  private formatPrice(price: string | number): string {
    const numPrice = typeof price === 'string' ? parseFloat(price) : price;
    if (isNaN(numPrice) || numPrice === 0) return 'price on request';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} crores`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} lakhs`;
    return `₹${numPrice.toLocaleString()}`;
  }

  private getGreetingResponse(): string {
    if (this.language === 'kn') {
      return kannadaResponses.GREETING;
    }
    return `Hello! I'm here to help you learn about this property. You can ask me about the price, location, amenities, room details, or anything else. What would you like to know?`;
  }

  private getPriceResponse(entities: Record<string, any> = {}): string {
    const price = this.formatPrice(this.property.sale_price || this.property.monthly_rent || '0');
    
    if (this.language === 'kn') {
      const negotiation = this.property.price_negotiable 
        ? negotiationText.yes
        : negotiationText.no;
      
      return formatKannadaResponse(kannadaResponses.PRICE_RESPONSE, {
        price,
        negotiation
      });
    }
    
    const negotiable = this.property.price_negotiable ? 'The owner is open to reasonable negotiations, especially for serious buyers.' : 'The price is fixed as it\'s already competitively priced.';
    
    return `This property is priced at ${price}. ${negotiable} The pricing includes all fixtures and fittings. Would you like to know about financing options or EMI details?`;
  }

  private getSizeResponse(entities: Record<string, any> = {}): string {
    const builtUp = this.property.built_up_area;
    const carpet = this.property.carpet_area;
    const area = builtUp || carpet;
    
    if (this.language === 'kn') {
      if (area) {
        return formatKannadaResponse(kannadaResponses.SIZE_RESPONSE, { area });
      }
      return 'ಆಸ್ತಿಯು ಉತ್ತಮ ಜಾಗ ಬಳಕೆಯನ್ನು ಹೊಂದಿದೆ. ನೀವು ಭೇಟಿಯ ಸಮಯದಲ್ಲಿ ಫ್ಲೋರ್ ಪ್ಲಾನ್ ನೋಡಬಹುದು.';
    }
    
    if (area) {
      return `This property has ${area} square feet of ${builtUp ? 'built-up' : 'carpet'} area. It's well-designed to maximize space utilization. The layout is very practical for daily living. Would you like to know about the room configuration?`;
    }
    
    return `The property has excellent space utilization with a practical layout. I can arrange for you to see the floor plan during a visit. Would you like to schedule a viewing?`;
  }

  private getLocationResponse(entities: Record<string, any> = {}): string {
    const locality = this.property.locality;
    const city = this.property.city;
    const nearby = this.property.nearby_places;
    
    if (this.language === 'kn') {
      const nearbyText = nearby && nearby.length > 0 
        ? formatNearbyInKannada(nearby)
        : '';
      
      return formatKannadaResponse(kannadaResponses.LOCATION_RESPONSE, {
        locality,
        city,
        nearby: nearbyText
      });
    }
    
    let response = `This property is located in ${locality}, ${city}, which is a well-connected area. `;
    
    if (nearby && nearby.length > 0) {
      response += `It's close to ${nearby.slice(0, 3).join(', ')}. `;
    }
    
    response += `The location offers great connectivity to major areas and has good infrastructure. Are you familiar with this locality?`;
    
    return response;
  }

  private getAmenitiesResponse(entities: Record<string, any> = {}): string {
    const amenities = this.property.amenities;
    
    if (this.language === 'kn') {
      if (amenities && amenities.length > 0) {
        const amenitiesText = translateAmenitiesToKannada(amenities.slice(0, 5));
        return formatKannadaResponse(kannadaResponses.AMENITIES_RESPONSE, {
          amenities: amenitiesText
        });
      }
      return 'ಈ ಆಸ್ತಿಯು ಎಲ್ಲಾ ಅತ್ಯಾವಶ್ಯಕ ಸೌಲಭ್ಯಗಳನ್ನು ಹೊಂದಿದೆ.';
    }
    
    if (amenities && amenities.length > 0) {
      const mainAmenities = amenities.slice(0, 5).join(', ');
      return `This property offers excellent amenities including ${mainAmenities}. The owner has ensured all modern facilities are available for comfortable living. Would you like to know about any specific amenity?`;
    }
    
    return `This property comes with all essential amenities for comfortable living. The owner has maintained everything very well. I can share the complete amenities list during our visit. Would you like to schedule a viewing?`;
  }

  private getRoomsResponse(entities: Record<string, any> = {}): string {
    const bedrooms = this.property.bedrooms;
    const bathrooms = this.property.bathrooms;
    const furnishing = this.property.furnishing;
    
    if (this.language === 'kn') {
      return formatKannadaResponse(kannadaResponses.ROOMS_RESPONSE, {
        bedrooms: bedrooms || 'N/A',
        bathrooms: bathrooms || 'N/A'
      });
    }
    
    let response = `This is a ${bedrooms ? `${bedrooms} bedroom` : 'well-designed'} property`;
    if (bathrooms) response += ` with ${bathrooms} bathroom${bathrooms > 1 ? 's' : ''}`;
    response += '. ';
    
    if (furnishing) {
      response += `It comes ${furnishing} furnished. `;
    }
    
    response += `The rooms are well-ventilated with good natural light. Would you like to know about the room sizes or layout?`;
    
    return response;
  }

  private getNegotiationResponse(): string {
    if (this.language === 'kn') {
      return this.property.price_negotiable 
        ? kannadaResponses.NEGOTIATION_RESPONSE_YES
        : kannadaResponses.NEGOTIATION_RESPONSE_NO;
    }
    
    if (this.property.price_negotiable) {
      return `Yes, the owner is open to reasonable negotiations. They understand market dynamics and are willing to discuss the price with serious buyers. The current pricing is already competitive, but there's some flexibility. Would you like to make an offer?`;
    }
    
    return `The owner has priced this property very competitively based on current market rates, so the price is fixed. However, they're open to discussing payment terms and other arrangements. Would you like to know about financing options?`;
  }

  private getConditionResponse(): string {
    const age = this.property.property_age;
    
    if (this.language === 'kn') {
      const ageText = age ? formatAgeInKannada(age) : '';
      return formatKannadaResponse(kannadaResponses.CONDITION_RESPONSE, { age: ageText });
    }
    
    let response = `This property is in excellent condition. `;
    
    if (age) {
      response += `It's ${age} old and has been very well maintained by the owner. `;
    }
    
    response += `All fixtures and fittings are in good working order. The owner has taken great care of this place. Would you like to schedule a visit to see the condition yourself?`;
    
    return response;
  }

  private getVisitResponse(): string {
    if (this.language === 'kn') {
      return kannadaResponses.VISIT_RESPONSE;
    }
    return `I'd be happy to arrange a visit for you! The owner is flexible with viewing times and wants serious buyers to see the property. You can schedule a convenient time, and I'll coordinate with the owner. The property shows much better in person. Would you like to book a visit now?`;
  }

  private getComparisonResponse(): string {
    if (this.language === 'kn') {
      return formatKannadaResponse(kannadaResponses.COMPARISON_RESPONSE, {
        bedrooms: this.property.bedrooms || 'N/A',
        locality: this.property.locality
      });
    }
    
    const bedrooms = this.property.bedrooms || 'N/A';
    const price = this.formatPrice(this.property.sale_price || this.property.monthly_rent || '0');
    const locality = this.property.locality;
    
    return `This ${bedrooms} bedroom property in ${locality} is priced at ${price}. To compare with similar properties in the area, I recommend checking the property listings page where you can filter by location and price range. This property offers good value for its features and location.`;
  }

  private getInvestmentResponse(): string {
    if (this.language === 'kn') {
      return formatKannadaResponse(kannadaResponses.INVESTMENT_RESPONSE, {
        locality: this.property.locality
      });
    }
    
    const locality = this.property.locality;
    const city = this.property.city;
    const price = this.formatPrice(this.property.sale_price || this.property.monthly_rent || '0');
    
    return `This property in ${locality}, ${city} is priced at ${price}. ${locality} is a well-established area with good infrastructure and connectivity. For detailed investment analysis including ROI projections, rental yield estimates, and market trends, I recommend consulting with our property advisors who can provide comprehensive market data for this specific location.`;
  }

  private getFurnishingResponse(): string {
    const furnishing = this.property.furnishing;
    
    if (this.language === 'kn') {
      if (furnishing) {
        const details = furnishingDetails[furnishing as keyof typeof furnishingDetails] || '';
        return formatKannadaResponse(kannadaResponses.FURNISHING_RESPONSE, {
          furnishing,
          details
        });
      }
      return 'ಆಸ್ತಿಯ ಸಜ್ಜು ವಿವರಗಳು ನಿಗದಿಪಡಿಸಿಲ್ಲ.';
    }
    
    if (furnishing) {
      return `This property comes ${furnishing} furnished. ${furnishing === 'fully' ? 'All necessary furniture and appliances are included, so you can move in immediately.' : furnishing === 'semi' ? 'Basic furniture is provided, and you can add your personal touches.' : 'The property is unfurnished, giving you complete freedom to design and furnish according to your preferences.'}`;
    }
    
    return `The furnishing details for this property are not specified. Please contact the owner or agent for more information about the furnishing status.`;
  }

  private getClarificationResponse(): string {
    if (this.language === 'kn') {
      return kannadaResponses.CLARIFICATION;
    }
    return `I'm not sure I understood that correctly. Could you please rephrase your question? You can ask me about the price, location, size, amenities, rooms, or any other aspect of this property.`;
  }

  private getDefaultResponse(): string {
    if (this.language === 'kn') {
      return kannadaResponses.DEFAULT;
    }
    return `That's a great question! I have detailed information about this property. You can ask me about the price, location, amenities, room details, or anything else you'd like to know. I'm here to help you make an informed decision. What specific aspect interests you most?`;
  }
}