// Property-specific intent classifier adapted from chatbot
// Runs entirely in frontend - no backend dependency

interface IntentResult {
  intent: string;
  confidence: number;
  entities: Record<string, any>;
}

export class PropertyIntentClassifier {
  private intentPatterns: Record<string, { keywords: string[]; patterns: RegExp[] }>;

  constructor() {
    // Property-specific intents only
    this.intentPatterns = {
      GREETING: {
        keywords: ['hello', 'hi', 'hey', 'good morning', 'good evening', 'namaste'],
        patterns: [
          /\b(hi|hello|hey|howdy)\b/i,
          /good (morning|evening|afternoon)/i,
          /\bnamaste\b/i
        ]
      },
      PRICE_QUERY: {
        keywords: ['price', 'cost', 'expensive', 'cheap', 'money', 'budget', 'worth'],
        patterns: [
          /\b(price|cost|expensive|cheap|money|budget)\b/i,
          /how much/i,
          /worth/i,
          /\baffordable\b/i
        ]
      },
      SIZE_QUERY: {
        keywords: ['size', 'area', 'space', 'square', 'feet', 'big', 'small', 'sqft'],
        patterns: [
          /\b(size|area|space|square|feet|sqft)\b/i,
          /how (big|large|small)/i,
          /\bcarpet area\b/i,
          /\bbuilt.?up\b/i
        ]
      },
      LOCATION_QUERY: {
        keywords: ['location', 'where', 'area', 'neighborhood', 'nearby', 'connectivity', 'transport'],
        patterns: [
          /\b(location|where|area|neighborhood|nearby)\b/i,
          /\b(connectivity|transport|metro|bus)\b/i,
          /how far/i
        ]
      },
      AMENITIES_QUERY: {
        keywords: ['amenities', 'facilities', 'features', 'gym', 'pool', 'parking', 'lift', 'security'],
        patterns: [
          /\b(amenities|facilities|features)\b/i,
          /\b(gym|pool|parking|lift|elevator|security)\b/i,
          /what (facilities|amenities)/i
        ]
      },
      ROOMS_QUERY: {
        keywords: ['bedroom', 'bathroom', 'room', 'bhk', 'toilet', 'bath'],
        patterns: [
          /\b(bedroom|bathroom|room|bhk|toilet|bath)\b/i,
          /how many (bedroom|bathroom|room)/i,
          /\d+\s*bhk/i
        ]
      },
      NEGOTIATION_QUERY: {
        keywords: ['negotiate', 'negotiable', 'discount', 'reduce', 'lower', 'bargain'],
        patterns: [
          /\b(negotiate|negotiable|discount|reduce|lower|bargain)\b/i,
          /can (i|we) negotiate/i,
          /\bflexible\b/i
        ]
      },
      CONDITION_QUERY: {
        keywords: ['age', 'old', 'new', 'condition', 'maintenance', 'renovation', 'repair'],
        patterns: [
          /\b(age|old|new|condition|maintenance|renovation)\b/i,
          /how old/i,
          /\brepair\b/i
        ]
      },
      VISIT_QUERY: {
        keywords: ['visit', 'see', 'viewing', 'tour', 'show', 'schedule', 'appointment'],
        patterns: [
          /\b(visit|see|viewing|tour|show|schedule|appointment)\b/i,
          /can (i|we) (visit|see)/i,
          /\bbook\b/i
        ]
      },
      COMPARISON_QUERY: {
        keywords: ['compare', 'comparison', 'similar', 'other', 'alternative', 'versus'],
        patterns: [
          /\b(compare|comparison|similar|other|alternative)\b/i,
          /\bvs\b|\bversus\b/i,
          /other properties/i
        ]
      },
      INVESTMENT_QUERY: {
        keywords: ['investment', 'invest', 'roi', 'return', 'profit', 'growth', 'appreciation'],
        patterns: [
          /\b(investment|invest|roi|return|profit)\b/i,
          /\b(growth|appreciation)\b/i,
          /good investment/i,
          /\brental yield\b/i
        ]
      },
      FURNISHING_QUERY: {
        keywords: ['furnish', 'furniture', 'furnished', 'unfurnished', 'semi'],
        patterns: [
          /\b(furnish|furniture|furnished|unfurnished)\b/i,
          /\bsemi.?furnished\b/i
        ]
      }
    };
  }

  classify(question: string): IntentResult {
    const q = question.toLowerCase().trim();
    const scores: Record<string, number> = {};

    // Calculate scores for each intent
    for (const [intent, config] of Object.entries(this.intentPatterns)) {
      let score = 0;

      // Keyword matching
      for (const keyword of config.keywords) {
        if (q.includes(keyword)) {
          score += 1;
        }
      }

      // Pattern matching (higher weight)
      for (const pattern of config.patterns) {
        if (pattern.test(q)) {
          score += 2;
        }
      }

      scores[intent] = score;
    }

    // Find best intent
    const maxScore = Math.max(...Object.values(scores));
    
    if (maxScore === 0) {
      return {
        intent: 'UNKNOWN',
        confidence: 0.3,
        entities: this.extractEntities(q)
      };
    }

    const bestIntent = Object.keys(scores).find(intent => scores[intent] === maxScore) || 'UNKNOWN';
    
    // Calculate confidence (0.5 to 0.95)
    const confidence = Math.min(0.95, 0.5 + (maxScore * 0.1));

    return {
      intent: bestIntent,
      confidence,
      entities: this.extractEntities(q)
    };
  }

  private extractEntities(text: string): Record<string, any> {
    const entities: Record<string, any> = {};

    // Extract BHK/Bedrooms
    const bhkMatch = text.match(/(\d+)\s*(bhk|bedroom|bed)/i);
    if (bhkMatch) {
      entities.bedrooms = bhkMatch[1];
    }

    // Extract specific amenities
    const amenities = ['gym', 'pool', 'parking', 'lift', 'elevator', 'security', 'garden', 'club'];
    for (const amenity of amenities) {
      if (text.includes(amenity)) {
        entities.amenity = amenity;
        break;
      }
    }

    // Extract room types
    if (text.includes('bathroom') || text.includes('toilet')) {
      entities.roomType = 'bathroom';
    } else if (text.includes('bedroom')) {
      entities.roomType = 'bedroom';
    }

    // Extract comparison context
    if (text.includes('similar') || text.includes('other') || text.includes('compare')) {
      entities.comparison = true;
    }

    // Extract question words for better context
    if (text.startsWith('how many')) {
      entities.questionType = 'count';
    } else if (text.startsWith('how much')) {
      entities.questionType = 'amount';
    } else if (text.startsWith('where')) {
      entities.questionType = 'location';
    } else if (text.startsWith('what')) {
      entities.questionType = 'description';
    } else if (text.startsWith('is') || text.startsWith('can')) {
      entities.questionType = 'boolean';
    }

    return entities;
  }
}

// Singleton instance
export const propertyIntentClassifier = new PropertyIntentClassifier();
