import { propertyViewsAPI } from '../api/propertyViews';
import { propertyViewsFallbackAPI } from '../api/propertyViewsFallback';

/**
 * Tracks property view using new dedicated view tracking system with fallback
 */
export const trackPropertyView = async (propertyId: number, userId?: string) => {
  try {
    // Only pass userId if user is actually logged in (has valid access token)
    const accessToken = localStorage.getItem('access_token');
    const validUserId = accessToken && userId ? userId : undefined;
    
    console.log('🔍 Starting view tracking:', {
      propertyId,
      originalUserId: userId,
      validUserId,
      hasAccessToken: !!accessToken,
      isAnonymous: !validUserId
    });
    
    // Try main API first
    let tracked = await propertyViewsAPI.trackView(propertyId, validUserId);
    
    // If main API fails and we have a valid user, try fallback
    if (!tracked && validUserId) {
      console.log('🔄 Main API failed, trying fallback...');
      tracked = await propertyViewsFallbackAPI.trackView(propertyId, validUserId);
    }
    
    if (tracked) {
      console.log(`✅ Property view tracked: ${propertyId} ${validUserId ? `(user: ${validUserId})` : '(anonymous)'}`);
    } else {
      console.log(`⏱️ Property view tracking failed: ${propertyId}`);
    }
    
    return tracked;
    
  } catch (error) {
    console.error('Error tracking property view:', error);
    return false;
  }
};

/**
 * Get property view statistics
 */
export const getPropertyViewStats = async (propertyId: number) => {
  try {
    return await propertyViewsAPI.getStats(propertyId);
  } catch (error) {
    console.error('Error getting property stats:', error);
    return null;
  }
};

/**
 * Get user's view history with fallback
 */
export const getUserViewHistory = async (userId: string, limit = 50) => {
  try {
    console.log('🔍 Getting user view history:', { userId, limit });
    
    // Use the working backend endpoint
    const result = await propertyViewsAPI.getUserHistory(userId, limit);
    
    console.log('📊 Final view history result:', result);
    return result;
  } catch (error) {
    console.error('Error getting user history:', error);
    return { views: [], count: 0 };
  }
};