import { userDashboardAPI } from '../api/user/userDashboard';

/**
 * Get view count for a specific property by analyzing user activities
 */
export const getPropertyViewCount = async (propertyId: number, userId: string): Promise<number> => {
  try {
    // Fetch all activities for the user
    const activities = await userDashboardAPI.getActivities(userId, 1, 1000);
    
    // Filter activities for property_view type and specific property
    const viewActivities = activities.filter(activity => 
      activity.activity_type === 'property_view' && 
      activity.property_id === propertyId
    );
    
    return viewActivities.length;
  } catch (error) {
    console.error('Error fetching property view count:', error);
    return 0;
  }
};

/**
 * Get view counts for multiple properties (for property owner dashboard)
 */
export const getMultiplePropertyViewCounts = async (propertyIds: number[], userId: string): Promise<Record<number, number>> => {
  try {
    const activities = await userDashboardAPI.getActivities(userId, 1, 1000);
    const viewCounts: Record<number, number> = {};
    
    // Initialize all property IDs with 0 views
    propertyIds.forEach(id => {
      viewCounts[id] = 0;
    });
    
    // Count views for each property
    activities.forEach(activity => {
      if (activity.activity_type === 'property_view' && propertyIds.includes(activity.property_id)) {
        viewCounts[activity.property_id] = (viewCounts[activity.property_id] || 0) + 1;
      }
    });
    
    return viewCounts;
  } catch (error) {
    console.error('Error fetching multiple property view counts:', error);
    return {};
  }
};

/**
 * Get total views across all properties for a user (property owner)
 */
export const getTotalPropertyViews = async (userId: string): Promise<number> => {
  try {
    const activities = await userDashboardAPI.getActivities(userId, 1, 1000);
    const viewActivities = activities.filter(activity => activity.activity_type === 'property_view');
    return viewActivities.length;
  } catch (error) {
    console.error('Error fetching total property views:', error);
    return 0;
  }
};