// Utility functions to safely access object properties and prevent undefined errors

export const safeGet = <T>(obj: any, path: string, defaultValue?: T): T | undefined => {
  try {
    const keys = path.split('.');
    let result = obj;
    
    for (const key of keys) {
      if (result === null || result === undefined) {
        return defaultValue;
      }
      result = result[key];
    }
    
    return result !== undefined ? result : defaultValue;
  } catch (error) {
    console.warn(`Safe access failed for path: ${path}`, error);
    return defaultValue;
  }
};

export const safeCall = <T>(fn: () => T, defaultValue?: T): T | undefined => {
  try {
    return fn();
  } catch (error) {
    console.warn('Safe call failed:', error);
    return defaultValue;
  }
};

export const safeLocalStorage = {
  getItem: (key: string, defaultValue?: string): string | null => {
    try {
      return localStorage.getItem(key) || defaultValue || null;
    } catch (error) {
      console.warn(`LocalStorage getItem failed for key: ${key}`, error);
      return defaultValue || null;
    }
  },
  
  setItem: (key: string, value: string): boolean => {
    try {
      localStorage.setItem(key, value);
      return true;
    } catch (error) {
      console.warn(`LocalStorage setItem failed for key: ${key}`, error);
      return false;
    }
  },
  
  removeItem: (key: string): boolean => {
    try {
      localStorage.removeItem(key);
      return true;
    } catch (error) {
      console.warn(`LocalStorage removeItem failed for key: ${key}`, error);
      return false;
    }
  }
};

export const isObjectValid = (obj: any): boolean => {
  return obj !== null && obj !== undefined && typeof obj === 'object';
};

export const hasProperty = (obj: any, prop: string): boolean => {
  return isObjectValid(obj) && Object.prototype.hasOwnProperty.call(obj, prop);
};