import crimeData from '../../crime-heatmap-india-major-cities.json';
 
export interface SafetyScore {
  score: number; // 0-5 scale
  intensity: number;
  zone: string;
  rating: 'Very Safe' | 'Safe' | 'Moderate' | 'Unsafe' | 'High Risk';
}
 
export const getSafetyScore = (lat: number, lng: number, city: string): SafetyScore | null => {
  // Find closest crime zone within 5km radius
  const nearbyZones = crimeData.filter(zone =>
    zone.city.toLowerCase() === city.toLowerCase() &&
    getDistance(lat, lng, zone.lat, zone.lon) <= 5
  );
 
  if (nearbyZones.length === 0) return null;
 
  // Get closest zone
  const closest = nearbyZones.reduce((prev, curr) =>
    getDistance(lat, lng, curr.lat, curr.lon) < getDistance(lat, lng, prev.lat, prev.lon)
      ? curr : prev
  );
 
  // Convert crime_score to safety score (0-5)
  // crime_score ranges from ~40-82 in dataset
  const minCrime = 40;
  const maxCrime = 82;
  const normalizedCrime = Math.max(0, Math.min(1, (closest.crime_score - minCrime) / (maxCrime - minCrime)));
  const safetyScore = (1 - normalizedCrime) * 5;
 
  return {
    score: Math.round(safetyScore * 10) / 10, // Round to 1 decimal place
    intensity: closest.crime_intensity,
    zone: closest.zone,
    rating: getRating(closest.crime_score)
  };
};
 
const getDistance = (lat1: number, lng1: number, lat2: number, lng2: number): number => {
  const R = 6371; // Earth's radius in km
  const dLat = (lat2 - lat1) * Math.PI / 180;
  const dLng = (lng2 - lng1) * Math.PI / 180;
  const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
            Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
            Math.sin(dLng/2) * Math.sin(dLng/2);
  return R * 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
};
 
const getRating = (crimeScore: number): SafetyScore['rating'] => {
  if (crimeScore <= 40) return 'Very Safe';
  if (crimeScore <= 55) return 'Safe';
  if (crimeScore <= 70) return 'Moderate';
  if (crimeScore <= 80) return 'Unsafe';
  return 'High Risk';
};