// Rate limiting
const rateLimiter = new Map<string, number[]>();

export const checkRateLimit = (userId: string, maxRequests = 10, windowMs = 60000): void => {
  const now = Date.now();
  const userRequests = rateLimiter.get(userId) || [];
  const recentRequests = userRequests.filter(time => now - time < windowMs);
  
  if (recentRequests.length >= maxRequests) {
    throw new Error('Rate limit exceeded. Try again later.');
  }
  
  rateLimiter.set(userId, [...recentRequests, now]);
};

// File content validation
export const validateFileContent = async (file: File): Promise<void> => {
  const buffer = await file.arrayBuffer();
  const header = new Uint8Array(buffer.slice(0, 8));
  
  const signatures: Record<string, number[]> = {
    'application/pdf': [0x25, 0x50, 0x44, 0x46],
    'image/jpeg': [0xFF, 0xD8, 0xFF],
    'image/png': [0x89, 0x50, 0x4E, 0x47]
  };
  
  const expectedSig = signatures[file.type];
  if (!expectedSig || !expectedSig.every((byte, i) => header[i] === byte)) {
    throw new Error('File content does not match declared type');
  }
};

// Session fingerprinting
let cachedFingerprint: string | null = null;

export const generateFingerprint = (): string => {
  if (cachedFingerprint) return cachedFingerprint;
  
  const canvas = document.createElement('canvas');
  const ctx = canvas.getContext('2d')!;
  ctx.textBaseline = 'top';
  ctx.font = '14px Arial';
  ctx.fillText('fp', 2, 2);
  
  const fp = canvas.toDataURL() + navigator.userAgent + screen.width + screen.height;
  cachedFingerprint = btoa(fp).slice(0, 32);
  return cachedFingerprint;
};

export const validateSession = (): void => {
  const stored = localStorage.getItem('session_fp');
  const current = generateFingerprint();
  
  if (!stored) {
    localStorage.setItem('session_fp', current);
    return;
  }
  
  if (stored !== current) {
    localStorage.clear();
    throw new Error('Session compromised');
  }
};

// State sanitization
export const sanitizeStateData = (state: any): any => {
  if (typeof state === 'string') {
    return state.replace(/[<>\"'&]/g, (m) => ({
      '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#x27;', '&': '&amp;'
    }[m] || m));
  }
  if (Array.isArray(state)) return state.map(sanitizeStateData);
  if (typeof state === 'object' && state !== null) {
    return Object.fromEntries(Object.entries(state).map(([k, v]) => [k, sanitizeStateData(v)]));
  }
  return state;
};