/**
 * RUTHLESS SESSION MANAGER - MEMORY-BASED SESSION CONTROL
 */

import { createRuthlessSession } from './sessionValidator';

/**
 * GENERATE UUID FOR DATABASE COMPATIBILITY
 */
const generateUUID = (): string => {
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
    const r = Math.random() * 16 | 0;
    const v = c === 'x' ? r : (r & 0x3 | 0x8);
    return v.toString(16);
  });
};

/**
 * INITIALIZE SESSION ON FIRST DOCUMENT UPLOAD
 */
export const initializeVerificationSession = (): string => {
  const existingSessionId = localStorage.getItem('verification-session-id');
  
  if (existingSessionId) {
    // Check if existing session is still valid
    const sessionData = localStorage.getItem(`session-${existingSessionId}`);
    if (sessionData) {
      try {
        const { expiresAt } = JSON.parse(sessionData);
        const remainingTime = new Date(expiresAt).getTime() - Date.now();
        
        if (remainingTime > 0) {
          console.log(`🔥 REUSING EXISTING SESSION: ${existingSessionId}`);
          return existingSessionId;
        }
      } catch (error) {
        console.warn('Invalid session data found, creating new session');
      }
    }
  }
  
  // Create new session with UUID format for database compatibility
  const newSessionId = generateUUID();
  
  // Create session with 24 hour expiration
  const now = Date.now();
  const expiresAt = new Date(now + (24 * 60 * 60 * 1000)); // 24 hours from now
  const sessionData = {
    sessionId: newSessionId,
    createdAt: new Date(now).toISOString(),
    expiresAt: expiresAt.toISOString(),
    userId: 'guest'
  };
  
  // Store session data
  localStorage.setItem('verification-session-id', newSessionId);
  localStorage.setItem('verification-session-timestamp', now.toString());
  localStorage.setItem(`session-${newSessionId}`, JSON.stringify(sessionData));
  
  console.log(`🔥 REUSING EXISTING SESSION: ${newSessionId}`);
  
  return newSessionId;
};

/**
 * GET CURRENT SESSION OR CREATE NEW ONE
 */
export const getCurrentSession = (): string | null => {
  const sessionId = localStorage.getItem('verification-session-id');
  
  if (!sessionId) return null;
  
  // Validate session exists and is not expired
  const sessionData = localStorage.getItem(`session-${sessionId}`);
  if (!sessionData) return null;
  
  try {
    const { expiresAt } = JSON.parse(sessionData);
    const remainingTime = new Date(expiresAt).getTime() - Date.now();
    
    if (remainingTime <= 0) {
      // Session expired - clean up
      localStorage.removeItem(`session-${sessionId}`);
      localStorage.removeItem('verification-session-id');
      return null;
    }
    
    return sessionId;
  } catch (error) {
    return null;
  }
};

/**
 * DESTROY CURRENT SESSION
 */
export const destroyCurrentSession = (): void => {
  const sessionId = localStorage.getItem('verification-session-id');
  
  if (sessionId) {
    localStorage.removeItem(`session-${sessionId}`);
  }
  
  localStorage.removeItem('verification-session-id');
  localStorage.removeItem('owner-details');
  localStorage.removeItem('verification-documents');
  localStorage.removeItem('ribil-score');
  localStorage.removeItem('verification-step');
  
  console.log('🔥 SESSION DESTROYED');
};