/**
 * Session validation utilities to prevent user session mixups
 */

export interface SessionInfo {
  userId: string;
  email: string;
  username: string;
  role: string;
  isValid: boolean;
}

/**
 * Validates the current session and returns user information
 */
export const validateCurrentSession = (): SessionInfo => {
  const defaultSession: SessionInfo = {
    userId: '',
    email: '',
    username: 'User',
    role: 'User',
    isValid: false
  };

  try {
    const accessToken = localStorage.getItem('access_token');
    
    if (!accessToken) {
      console.warn('No access token found');
      return defaultSession;
    }

    // Decode JWT token
    const payload = JSON.parse(atob(accessToken.split('.')[1]));
    
    // Check if token is expired
    if (!payload.exp || payload.exp <= Date.now() / 1000) {
      console.warn('Access token is expired');
      clearAuthData();
      return defaultSession;
    }

    // Get user data from localStorage
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const storedEmail = localStorage.getItem('login_email') || localStorage.getItem('user_email') || user.email || '';

    // Validate that token user_id matches stored user data (if both exist)
    const tokenUserId = payload.user_id || payload.id || payload.sub;
    const storedUserId = user.user_id || user.id;

    // Only validate mismatch if both IDs exist and are different
    if (tokenUserId && storedUserId && tokenUserId !== storedUserId) {
      console.error('User ID mismatch detected:', { tokenUserId, storedUserId });
      clearAuthData();
      return defaultSession;
    }

    // Extract username
    let username = 'User';
    if (user.first_name) {
      username = user.first_name;
    } else if (storedEmail) {
      const emailPart = storedEmail.split('@')[0];
      username = emailPart.replace(/[0-9]/g, '').replace(/[^a-zA-Z]/g, '') || emailPart;
    }

    // Determine role from multiple possible sources
    let role = 'User';
    if (payload.is_admin === true) {
      role = 'Admin';
    } else if (payload.roles && Array.isArray(payload.roles)) {
      if (payload.roles.includes('Admin')) role = 'Admin';
      else if (payload.roles.includes('Agent')) role = 'Agent';
      else if (payload.roles.includes('Seller')) role = 'Seller';
      else if (payload.roles.includes('Buyer')) role = 'Buyer';
    } else if (user.role) {
      role = user.role;
    }

    return {
      userId: tokenUserId || storedUserId || '',
      email: storedEmail,
      username,
      role,
      isValid: true
    };

  } catch (error) {
    console.error('Error validating session:', error);
    clearAuthData();
    return defaultSession;
  }
};

/**
 * Clears all authentication data
 */
export const clearAuthData = (): void => {
  const keysToRemove = [
    'access_token',
    'refresh_token',
    'user',
    'login_email',
    'user_email',
    'adminLoggedIn',
    'sellerToken',
    'agentToken'
  ];

  keysToRemove.forEach(key => localStorage.removeItem(key));
  sessionStorage.clear();
  
  console.log('Authentication data cleared');
};

/**
 * Checks if the current session belongs to the expected user
 */
export const validateUserSession = (expectedEmail?: string): boolean => {
  const session = validateCurrentSession();
  
  if (!session.isValid) {
    return false;
  }

  // If expected email is provided, validate it matches
  if (expectedEmail && session.email !== expectedEmail) {
    console.error('Email mismatch detected:', { expected: expectedEmail, actual: session.email });
    clearAuthData();
    return false;
  }

  return true;
};

/**
 * Gets the current user's email from the most reliable source
 */
export const getCurrentUserEmail = (): string => {
  const session = validateCurrentSession();
  return session.email;
};

/**
 * Gets the current user's ID from the JWT token
 */
export const getCurrentUserId = (): string => {
  const session = validateCurrentSession();
  return session.userId;
};

/**
 * Creates a ruthless verification session
 */
export const createRuthlessSession = (sessionId: string): void => {
  const now = Date.now();
  const expiresAt = new Date(now + (24 * 60 * 60 * 1000)); // 24 hours from now
  
  const sessionData = {
    sessionId,
    createdAt: new Date(now).toISOString(),
    expiresAt: expiresAt.toISOString(),
    userId: getCurrentUserId()
  };
  
  // Store session data
  localStorage.setItem('verification-session-id', sessionId);
  localStorage.setItem('verification-session-timestamp', now.toString());
  localStorage.setItem(`session-${sessionId}`, JSON.stringify(sessionData));
  
  console.log(`🔥 RUTHLESS SESSION CREATED: ${sessionId}`);
};

/**
 * Ruthless session validator for verification processes
 */
export const ruthlessSessionValidator = {
  /**
   * Validates a verification session
   */
  validateSession: async (sessionId: string, userId: string) => {
    try {
      // Check if session exists in localStorage
      const storedSessionId = localStorage.getItem('verification-session-id');
      
      if (!storedSessionId || storedSessionId !== sessionId) {
        return {
          valid: false,
          expired: true,
          remainingTime: 0
        };
      }
      
      // Check session data
      const sessionData = localStorage.getItem(`session-${sessionId}`);
      if (!sessionData) {
        return {
          valid: false,
          expired: true,
          remainingTime: 0
        };
      }
      
      try {
        const session = JSON.parse(sessionData);
        const now = Date.now();
        const expiresAt = new Date(session.expiresAt).getTime();
        const remainingTime = expiresAt - now;
        
        if (remainingTime <= 0) {
          return {
            valid: false,
            expired: true,
            remainingTime: 0
          };
        }
        
        return {
          valid: true,
          expired: false,
          remainingTime
        };
      } catch (parseError) {
        console.error('Error parsing session data:', parseError);
        return {
          valid: false,
          expired: true,
          remainingTime: 0
        };
      }
      
    } catch (error) {
      console.error('Session validation error:', error);
      return {
        valid: false,
        expired: true,
        remainingTime: 0
      };
    }
  }
};