/**
 * Test utility to simulate session expiration for testing purposes
 */

export const simulateSessionExpiration = () => {
  // Get current session ID
  const sessionId = localStorage.getItem('verification-session-id');
  
  if (sessionId) {
    // Create expired session data
    const expiredSessionData = {
      createdAt: new Date(Date.now() - 25 * 60 * 60 * 1000).toISOString(), // 25 hours ago
      expiresAt: new Date(Date.now() - 1 * 60 * 60 * 1000).toISOString(), // 1 hour ago (expired)
      sessionId
    };
    
    // Update localStorage with expired session
    localStorage.setItem(`session-${sessionId}`, JSON.stringify(expiredSessionData));
    
    console.log('🔥 Session expiration simulated - session should be detected as expired');
    console.log('Expired session data:', expiredSessionData);
    
    return true;
  }
  
  console.warn('No session ID found to expire');
  return false;
};

export const simulateNearExpiration = (minutesRemaining: number = 5) => {
  // Get current session ID
  const sessionId = localStorage.getItem('verification-session-id');
  
  if (sessionId) {
    const now = Date.now();
    const expiresAt = new Date(now + minutesRemaining * 60 * 1000).toISOString(); // Expires in specified minutes
    
    // Create near-expiration session data
    const nearExpirationData = {
      createdAt: new Date(now - (24 * 60 * 60 * 1000 - minutesRemaining * 60 * 1000)).toISOString(),
      expiresAt,
      sessionId
    };
    
    // Update localStorage with near-expiration session
    localStorage.setItem(`session-${sessionId}`, JSON.stringify(nearExpirationData));
    
    console.log(`🔥 Session near-expiration simulated - expires in ${minutesRemaining} minutes`);
    console.log('Near-expiration session data:', nearExpirationData);
    
    return true;
  }
  
  console.warn('No session ID found to modify');
  return false;
};

export const resetSessionToValid = () => {
  // Get current session ID
  const sessionId = localStorage.getItem('verification-session-id');
  
  if (sessionId) {
    const now = Date.now();
    const expiresAt = new Date(now + 24 * 60 * 60 * 1000).toISOString(); // 24 hours from now
    
    // Create valid session data
    const validSessionData = {
      createdAt: new Date(now).toISOString(),
      expiresAt,
      sessionId
    };
    
    // Update localStorage with valid session
    localStorage.setItem(`session-${sessionId}`, JSON.stringify(validSessionData));
    
    console.log('✅ Session reset to valid state');
    console.log('Valid session data:', validSessionData);
    
    return true;
  }
  
  console.warn('No session ID found to reset');
  return false;
};

// Make functions available globally for testing in browser console
if (typeof window !== 'undefined') {
  (window as any).testSessionExpiration = {
    simulateExpiration: simulateSessionExpiration,
    simulateNearExpiration: simulateNearExpiration,
    resetToValid: resetSessionToValid
  };
  
  console.log('🧪 Session testing utilities loaded. Use:');
  console.log('- window.testSessionExpiration.simulateExpiration() - to simulate expired session');
  console.log('- window.testSessionExpiration.simulateNearExpiration(5) - to simulate session expiring in 5 minutes');
  console.log('- window.testSessionExpiration.resetToValid() - to reset session to valid state');
}