// Utility functions to add event tracking to existing components
// Import this in components that need tracking

import { behaviorTracker } from './behaviorTracker';

/**
 * Track image click event
 * Call this when user clicks on property images
 */
export const trackImageClick = (propertyId: number, imageIndex?: number, imageUrl?: string) => {
  behaviorTracker.trackImageClick(propertyId, {
    image_index: imageIndex,
    image_url: imageUrl
  });
};

/**
 * Track document click event
 * Call this when user views/downloads property documents
 */
export const trackDocumentClick = (propertyId: number, documentType?: string, documentName?: string) => {
  behaviorTracker.trackDocumentClick(propertyId, {
    document_type: documentType,
    document_name: documentName
  });
};

/**
 * Track schedule visit event
 * Call this when user schedules a property visit
 */
export const trackScheduleVisit = (propertyId: number, visitDate?: string, visitTime?: string) => {
  behaviorTracker.trackScheduleVisit(propertyId, {
    visit_date: visitDate,
    visit_time: visitTime
  });
};

/**
 * Track bid placement event
 * Call this when user places a bid on auction property
 */
export const trackBidPlace = (propertyId: number, bidAmount?: number) => {
  behaviorTracker.trackBidPlace(propertyId, {
    bid_amount: bidAmount
  });
};

/**
 * Track auction registration event
 * Call this when user registers for an auction
 */
export const trackAuctionRegister = (propertyId: number) => {
  behaviorTracker.trackAuctionRegister(propertyId, {
    registration_time: new Date().toISOString()
  });
};

// Example usage in components:
/*
// In image gallery component:
<img 
  src={imageUrl} 
  onClick={() => trackImageClick(propertyId, index, imageUrl)}
/>

// In document viewer:
<button onClick={() => trackDocumentClick(propertyId, 'pdf', 'property_deed.pdf')}>
  View Document
</button>

// In visit scheduling form:
<button onClick={() => trackScheduleVisit(propertyId, selectedDate, selectedTime)}>
  Schedule Visit
</button>

// In bidding component:
<button onClick={() => trackBidPlace(propertyId, bidAmount)}>
  Place Bid
</button>

// In auction registration:
<button onClick={() => trackAuctionRegister(propertyId)}>
  Register for Auction
</button>
*/
