import { kannadaToEnglish, englishToKannada } from './kannadaTranslations';

/**
 * Translate Kannada text to English for intent classification
 */
export function translateKannadaToEnglish(text: string): string {
  if (!text || text.trim().length === 0) return '';
  
  const words = text.toLowerCase().trim().split(/\s+/);
  const translatedWords: string[] = [];
  
  for (const word of words) {
    // Remove punctuation
    const cleanWord = word.replace(/[.,!?;:]/g, '');
    
    // Look up in dictionary
    if (kannadaToEnglish[cleanWord]) {
      translatedWords.push(kannadaToEnglish[cleanWord]);
    } else {
      // Keep original if not found (might be English or proper noun)
      translatedWords.push(cleanWord);
    }
  }
  
  return translatedWords.join(' ');
}

/**
 * Detect if text contains Kannada characters
 */
export function isKannada(text: string): boolean {
  // Kannada Unicode range: 0C80-0CFF
  const kannadaRegex = /[\u0C80-\u0CFF]/;
  return kannadaRegex.test(text);
}

/**
 * Format Kannada response by replacing variables
 */
export function formatKannadaResponse(
  template: string,
  variables: Record<string, string | number>
): string {
  let response = template;
  
  for (const [key, value] of Object.entries(variables)) {
    const placeholder = `{${key}}`;
    response = response.replace(new RegExp(placeholder, 'g'), String(value));
  }
  
  return response;
}

/**
 * Translate amenities list to Kannada
 */
export function translateAmenitiesToKannada(amenities: string[]): string {
  if (!amenities || amenities.length === 0) return '';
  
  const translated = amenities.map(amenity => {
    const lower = amenity.toLowerCase();
    return englishToKannada[lower] || amenity;
  });
  
  return translated.join(', ');
}

/**
 * Format property age in Kannada
 */
export function formatAgeInKannada(age: string): string {
  if (!age) return '';
  
  // Extract number from age string
  const match = age.match(/\d+/);
  if (match) {
    const years = match[0];
    return `ಇದು ${years} ವರ್ಷ ಹಳೆಯದು`;
  }
  
  return age;
}

/**
 * Format nearby places in Kannada
 */
export function formatNearbyInKannada(places: string[]): string {
  if (!places || places.length === 0) return '';
  
  const placesText = places.slice(0, 3).join(', ');
  return `ಇದು ${placesText} ಹತ್ತಿರದಲ್ಲಿದೆ`;
}
