// Urban Growth Calculation Utilities

import { UrbanGrowthData } from '../types/gee';

/**
 * Calculate urban growth score based on expansion rate and development
 */
export function calculateGrowthScore(
  growthRate: number,
  infrastructureScore: number,
  builtUpDensity: number
): number {
  // Growth rate weight: 50%
  const growthComponent = Math.min(growthRate / 10, 1) * 5;
  
  // Infrastructure weight: 30%
  const infraComponent = (infrastructureScore / 10) * 3;
  
  // Density weight: 20%
  const densityComponent = (builtUpDensity / 100) * 2;
  
  return Math.min(10, growthComponent + infraComponent + densityComponent);
}

/**
 * Classify development level based on growth patterns
 */
export function classifyDevelopmentLevel(
  growthRate: number,
  builtUpDensity: number
): 'emerging' | 'developing' | 'mature' | 'saturated' {
  if (growthRate > 8 && builtUpDensity < 40) return 'emerging';
  if (growthRate > 4 && builtUpDensity < 70) return 'developing';
  if (growthRate > 1 && builtUpDensity < 90) return 'mature';
  return 'saturated';
}

/**
 * Calculate investment potential score
 */
export function calculateInvestmentPotential(
  growthRate: number,
  infrastructureScore: number,
  currentDensity: number
): number {
  // High growth + good infrastructure + room to grow = high potential
  const growthFactor = Math.min(growthRate / 10, 1) * 4;
  const infraFactor = (infrastructureScore / 10) * 3;
  const spaceFactor = ((100 - currentDensity) / 100) * 3;
  
  return Math.min(10, growthFactor + infraFactor + spaceFactor);
}

/**
 * Get color for growth level
 */
export function getGrowthColor(score: number): string {
  if (score >= 7) return '#22c55e'; // green - high growth
  if (score >= 4) return '#3b82f6'; // blue - moderate growth
  if (score >= 2) return '#f59e0b'; // yellow - slow growth
  return '#ef4444'; // red - declining
}

/**
 * Get background color for growth badge
 */
export function getGrowthBgColor(level: string): string {
  if (level === 'emerging') return 'bg-green-100 text-green-800 border-green-300';
  if (level === 'developing') return 'bg-blue-100 text-blue-800 border-blue-300';
  if (level === 'mature') return 'bg-yellow-100 text-yellow-800 border-yellow-300';
  return 'bg-gray-100 text-gray-800 border-gray-300';
}

/**
 * Format urban growth data
 */
export function formatUrbanGrowthData(
  growthRate: number,
  infrastructureScore: number,
  builtUpDensity: number,
  timelineData: Array<{ year: number; urbanArea: number; builtUpDensity: number }>
): UrbanGrowthData {
  const growthScore = calculateGrowthScore(growthRate, infrastructureScore, builtUpDensity);
  const developmentLevel = classifyDevelopmentLevel(growthRate, builtUpDensity);
  const investmentPotential = calculateInvestmentPotential(growthRate, infrastructureScore, builtUpDensity);
  
  // Calculate urban area change from timeline
  const urbanAreaChange = timelineData.length >= 2
    ? ((timelineData[timelineData.length - 1].urbanArea - timelineData[0].urbanArea) / timelineData[0].urbanArea) * 100
    : 0;
  
  return {
    growthScore,
    growthRate,
    developmentLevel,
    urbanAreaChange,
    infrastructureScore,
    investmentPotential,
    timelineData,
    lastUpdated: new Date().toISOString()
  };
}

/**
 * Generate cache key for urban growth data
 */
export function getGrowthCacheKey(lat: number, lng: number): string {
  return `growth_${lat.toFixed(4)}_${lng.toFixed(4)}`;
}
