// Centralized user ID extraction utility
export const getUserId = (): string | null => {
  try {
    // Debug: Log what's in localStorage
    console.log('🔍 getUserId: Checking localStorage...');
    
    // Check multiple possible user storage locations
    const userStr = localStorage.getItem('user');
    const agentId = localStorage.getItem('agentId');
    const sellerId = localStorage.getItem('sellerId');
    const userId = localStorage.getItem('userId');
    const currentUserId = localStorage.getItem('current_user_id');
    
    console.log('🔍 getUserId: Storage check:', {
      user: userStr,
      agentId,
      sellerId,
      userId,
      currentUserId
    });
    
    // Try direct ID storage first
    if (agentId) {
      console.log('✅ getUserId: Using agentId:', agentId);
      return agentId;
    }
    
    if (sellerId) {
      console.log('✅ getUserId: Using sellerId:', sellerId);
      return sellerId;
    }
    
    if (userId) {
      console.log('✅ getUserId: Using userId:', userId);
      return userId;
    }
    
    if (currentUserId) {
      console.log('✅ getUserId: Using currentUserId:', currentUserId);
      return currentUserId;
    }
    
    // Try parsing user object
    if (userStr && userStr !== '{}' && userStr !== 'null') {
      const user = JSON.parse(userStr);
      console.log('🔍 getUserId: Parsed user object:', user);
      
      // Try multiple possible user ID fields
      const extractedUserId = user.id || user.user_id || user.uuid || user.userId;
      if (extractedUserId) {
        console.log('✅ getUserId: Extracted from user object:', extractedUserId);
        return extractedUserId;
      }
    }
    
    console.warn('⚠️ getUserId: No user ID found in any storage location');
    return null;
  } catch (error) {
    console.error('❌ getUserId: Error parsing user data:', error);
    return null;
  }
};

const isUuid = (value: string | null): boolean => {
  if (!value) return false;
  return /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i.test(value);
};

export const getUserUuidFromToken = (): string | null => {
  try {
    const token =
      localStorage.getItem('sellerToken') ||
      localStorage.getItem('access_token') ||
      localStorage.getItem('agentToken') ||
      localStorage.getItem('token') ||
      localStorage.getItem('auth_token');

    if (!token || !token.includes('.')) return null;
    const payload = JSON.parse(atob(token.split('.')[1]));
    const candidate = payload.user_id || payload.userId || payload.sub || payload.id;
    return isUuid(candidate) ? candidate : null;
  } catch (error) {
    console.warn('⚠️ getUserUuidFromToken: Failed to decode token', error);
    return null;
  }
};

export const getCurrentUserUuid = (): string | null => {
  const sellerId = localStorage.getItem('sellerId');
  const userId = localStorage.getItem('user_id');
  if (isUuid(sellerId)) return sellerId;
  if (isUuid(userId)) return userId;

  const tokenUuid = getUserUuidFromToken();
  if (tokenUuid) {
    localStorage.setItem('user_id', tokenUuid);
    if (!sellerId) {
      localStorage.setItem('sellerId', tokenUuid);
    }
    return tokenUuid;
  }

  return null;
};

export const validateUserAuthentication = (): string => {
  const userId = getUserId();
  
  if (!userId) {
    // Don't throw error, just return a temporary ID and log warning
    console.warn('⚠️ validateUserAuthentication: No authenticated user found, using temporary ID');
    return 'temp_user_' + Date.now();
  }
  
  return userId;
};

// Get or create a session user ID for property creation
export const getSessionUserId = (): string => {
  // First try to get authenticated user ID
  const userId = getUserId();
  
  if (userId) {
    return userId;
  }
  
  // If no authenticated user, create/get a session-based ID
  let sessionUserId = localStorage.getItem('session_user_id');
  
  if (!sessionUserId) {
    // Create a new session user ID
    sessionUserId = 'session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    localStorage.setItem('session_user_id', sessionUserId);
    console.log('🆔 Created new session user ID:', sessionUserId);
  } else {
    console.log('🆔 Using existing session user ID:', sessionUserId);
  }
  
  return sessionUserId;
};
