import { visitScheduleAPI } from '../api/user/visitScheduleApi';
import { getUserId } from './userUtils';
import { toast } from 'sonner';

/**
 * Check if user has already scheduled a visit for a specific property
 */
export const checkDuplicateVisit = async (propertyId: number): Promise<boolean> => {
  try {
    const userId = getUserId();
    if (!userId) {
      console.log('No userId found');
      return false;
    }
    
    const response = await visitScheduleAPI.getUserVisitSchedules(userId, 1, 100);
    console.log('User visits:', response.visits);
    console.log('Checking for propertyId:', propertyId);
    
    const hasDuplicate = response.visits.some(visit => visit.property_id === propertyId);
    console.log('Has duplicate visit:', hasDuplicate);
    
    return hasDuplicate;
  } catch (error) {
    console.error('Error checking duplicate visit:', error);
    return false;
  }
};

/**
 * Validate visit scheduling and show appropriate toast message
 */
export const validateVisitScheduling = async (
  propertyId: number, 
  propertyOwnerEmail?: string
): Promise<{ canSchedule: boolean; reason?: string }> => {
  // Check if user is property owner
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  const userEmail = user.email;
  const isOwner = userEmail && propertyOwnerEmail && 
    userEmail.toLowerCase() === propertyOwnerEmail.toLowerCase();
  
  if (isOwner) {
    toast.error("You cannot schedule a visit to your own property.");
    return { canSchedule: false, reason: 'owner' };
  }
  
  // Check for duplicate visit
  const hasDuplicate = await checkDuplicateVisit(propertyId);
  console.log('Validation - hasDuplicate:', hasDuplicate);
  
  if (hasDuplicate) {
    console.log('Showing duplicate visit toast');
    toast.error("You've already scheduled a visit for this property.");
    return { canSchedule: false, reason: 'duplicate' };
  }
  
  return { canSchedule: true };
};